<?php

namespace App\DataTables;

use App\Models\Currency;
use App\Models\LedgerTransfer;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class AdvancedSearchDataTable extends DataTable
{
  protected $filters = [];

  public function __construct($filters = [])
  {
    $this->filters = $filters;
  }

  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      // ->setRowClass(function ($query) {
      //     return $query->status === 'completed' ? 'table-success' : ($query->status === 'pending' ? 'table-warning' : 'table-danger');
      // })
      ->addColumn('action', function ($data) {
        return view('Transfers.actions', compact('data'));
      })
      ->addColumn("from", function ($query) {
        return getAgentHtml($query->sender) ?? 'N/A';
      })
      ->addColumn("to", function ($query) {
        return getAgentHtml($query->receiver) ?? 'N/A';
      })
      ->addColumn("debit_entry", function ($query) {
        $html = "";
        $amount = $query->amount;
        $currency = Currency::getCurrencyName($query->currency);
        $color = Currency::getCurrencyColor($query->currency);
        $fee = $query->send_fee;
        $html .= "<span>" . PriceFormat($amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:blue'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })
      ->addColumn("credit_entry", function ($query) {
        $html = "";
        $amount = $query->amount;
        $currency = Currency::getCurrencyName($query->delivery_currency);
        $color = Currency::getCurrencyColor($query->delivery_currency);
        $fee = $query->receiver_fee ?? 0;
        $html .= "<span>" . PriceFormat($amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:red'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })
      ->addColumn("amount", function ($query) {
        return PriceFormat($query->amount);
      })
      ->editColumn("status", function ($query) {
        return "<span class='badge bg-" . $query->status_color . "'>" . __($query->status) . "</span>";
      })
      ->addColumn("created_by", function ($query) {
        return $query->user->name ?? 'N/A';
      })
      ->addColumn("created_at", function ($query) {
        return $query->created_at->format('Y-m-d H:i');
      })
      ->editColumn("note", function ($query) {
        return $query->note ?? __("general.no_note");
      })
      ->addColumn("profit", function ($query) {
        return PriceFormat($query->send_fee - $query->receiver_fee);
      })
      ->addColumn("customer", function ($query) {
        return $query->recipient_name ?? 'N/A';
      })

      ->filterColumn('from', function ($query, $keyword) {
        $query->whereHas('sender', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('to', function ($query, $keyword) {
        $query->whereHas('receiver', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('customer', function ($query, $keyword) {
        $query->where('recipient_name', 'like', "%{$keyword}%");
      })
      ->filterColumn('note', function ($query, $keyword) {
        $query->where('notes', 'like', "%{$keyword}%");
      })
      ->filterColumn('debit_entry', function ($query, $keyword) {
        $query->where('amount', 'like', "%{$keyword}%");
      })
      ->filterColumn('status', function ($query, $keyword) {
        $query->where('status', 'like', "%{$keyword}%");
      })
      ->filterColumn('created_at', function ($query, $keyword) {
        $query->where('created_at', 'like', "%{$keyword}%");
      })
      ->filterColumn('created_by', function ($query, $keyword) {
        $query->whereHas('user', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('reference', function ($query, $keyword) {
        $query->where('reference', 'like', "%{$keyword}%");
      })

      ->orderColumn('reference', function ($query, $order) {
        $query->orderBy('reference', $order);
      })
      ->orderColumn('from', function ($query, $order) {
        $query->orderBy('sender_id', $order);
      })
      ->orderColumn('to', function ($query, $order) {
        $query->orderBy('receiver_id', $order);
      })
      ->orderColumn('customer', function ($query, $order) {
        $query->orderBy('recipient_name', $order);
      })
      ->orderColumn('debit_entry', function ($query, $order) {
        $query->orderBy('amount', $order);
      })
      ->orderColumn('credit_entry', function ($query, $order) {
        $query->orderBy('delivery_amount', $order);
      })
      ->orderColumn('profit', function ($query, $order) {
        $query->orderByRaw("(send_fee - receiver_fee) {$order}");
      })
      ->orderColumn('created_by', function ($query, $order) {
        $query->orderBy('user_id', $order);
      })
      ->orderColumn('status', function ($query, $order) {
        $query->orderBy('status', $order);
      })
      ->orderColumn('notes', function ($query, $order) {
        $query->orderBy('notes', $order);
      })
      ->orderColumn('created_at', function ($query, $order) {
        $query->orderBy('created_at', $order);
      })
      ->rawColumns(['action', 'debit_entry', 'credit_entry', 'amount', 'status', 'created_at', 'note', 'from', 'to'])
      ->setRowId('id');
  }

  public function query(LedgerTransfer $model): QueryBuilder
  {
    $query = $model->newQuery()->withoutGlobalScopes()
      ->with([
        'debitJournalEntry.details',
        'creditJournalEntry.details',
        'sender',
        'receiver'
      ]);

    if (auth()->guard("agent")->check()) {
      $agent = auth()->guard("agent")->user()->agent;
      $root = $agent->rootAgent();
      $childs = $root->getallchildid();
      $query->whereIn("sender_id", $childs)->orWhereIn("receiver_id", $childs);
    }
    // Get filters from request if not passed to constructor
    $filters = $this->filters ?: request()->get('filters', []);

    // If no filters are applied, return empty result set
    if (empty($filters)) {
      return $query->whereRaw('1 = 0');
    }

    // Apply filters if any exist
    return $model->Filter($filters);
  }

  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'processing' => true,
        'serverSide' => true,
        'deferRender' => true,
        'scrollX' => true,
        'pageLength' => 10,
        'lengthMenu' => [10, 25, 50, 100],
        'initComplete' => 'function() {
                    // Show message when no filters applied
                    if (Object.keys(window.filters || {}).length === 0) {
                        $("#transfer-table").DataTable().$("td").html("<div class=\'text-center py-4\'>' . __('transfers.apply_filters_to_see_results') . '</div>");
                    }
                }',
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])
      ->setTableId('advancedsearch-table')
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
                    const urlParams = new URLSearchParams(window.location.search);
                    const show = urlParams.get("show");
                    data.filters = window.filters || {};
                    if (show) {
                        data.filters.show = show;
                    };
                    data.search_term = $("#general-search").val();
                }'
      ])
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>r' .
        't' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')

      ->orderBy(0, 'desc')
      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_agent"),
          'className' => 'btn btn-primary ms-2 new_transfer_btn',
          'action' => 'function (e, dt, node, config) { window.location = "/Transfers/create"; }',
          'attr' => [
            'title' => __("agents.add_agent"),
          ],
        ]),
        Button::raw([
          'text' => '<i class="ti ti-search"></i> ',
          'className' => 'btn btn-info ms-2',
          'attr' => [
            'title' => __("general.search"),
            'data-bs-toggle' => 'offcanvas',
            'data-bs-target' => '#FilterModal',
          ],
        ])
      ]);
  }

  public function getColumns(): array
  {
    return [
      Column::make("created_at")
        ->title(__('transfers.created_at'))
        ->orderable(true)
        ->searchable(true)
        ->width('10%'),
      Column::make("debit_entry")
        ->title(__('transfers.debit'))
        ->orderable(true)
        ->searchable(true)
        ->width('10%')
        ->searchable(false),
      Column::make("from")
        ->title(__('transfers.from'))
        ->orderable(true)
        ->searchable(true)
        ->width('12%'),
      Column::make("customer")
        ->title(__('transfers.receiver_client'))
        ->orderable(true)
        ->searchable(true)
        ->width('12%'),
      Column::make("credit_entry")
        ->title(__('transfers.credit'))
        ->orderable(true)
        ->searchable(true)
        ->width('10%')
        ->searchable(false),
      Column::make("to")
        ->title(__('transfers.to'))
        ->orderable(true)
        ->searchable(true)
        ->width('12%'),
      Column::make("profit")
        ->title(__('transfers.profit'))
        ->orderable(true)
        ->searchable(true)
        ->width('5%'),
      Column::make("notes")
        ->title(__('transfers.notes'))
        ->orderable(true)
        ->searchable(true)
        ->width('10%'),
      Column::make("created_by")
        ->title(__('transfers.created_by'))
        ->orderable(true)
        ->searchable(true)
        ->width('10%'),
      Column::make("reference")
        ->title(__('transfers.reference'))
        ->orderable(true)
        ->searchable(true)
        ->width('20%'),
      Column::make("status")
        ->title(__('transfers.status'))
        ->orderable(true)
        ->searchable(true)
        ->width('20%'),
      Column::computed('action')
        ->title(__('Actions'))
        ->exportable(false)
        ->printable(false)
        ->width('6%')
        ->addClass('text-center'),
    ];
  }

  protected function filename(): string
  {
    return 'Transfer_' . date('YmdHis');
  }
}
