<?php

namespace App\DataTables;

use App\Models\User;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class AgentUsersDataTable extends DataTable
{
  protected $filters;
  public function __construct($filters = [])
  {
    $this->filters = $filters;
  }
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('action', function ($data) {
        $type = "actions";
        return view('User.actions', compact('data', 'type'))->render();
      })
      ->addColumn("role", function ($query) {
        return  $query->getRoleNames()->first();
      })
      ->addColumn("status", function ($query) {
        $data = $query;
        $type = "status";
        return view('User.actions', compact('data', 'type'))->render();
      })
      ->addColumn("2fa", function ($query) {
        $data = $query;
        $type = "2fa";
        return view('User.actions', compact('data', 'type'))->render();
      })
      ->editColumn("gender", function ($query) {
        return   __("users." . $query->gender);
      })
      ->editColumn("company", function ($query) {
        return   $query->company->name ?? 'N/A';
      })
      ->rawColumns(['action', 'status', '2fa'])
      ->setRowId('id');
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(User $model): QueryBuilder
  {
    $query = $model->newQuery()->where('agent_id', '!=', null);
    if (isset($this->filters['agent_id'])) {
      $query->where('agent_id', $this->filters['agent_id']);
    }
    return $query;
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('client-table')
      ->parameters(
        [
          'language' => [
            'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
          ],
          'processing' => true,
          'serverSide' => true,
          'rowReorder' => [
            'selector' => 'tr>td:not(:last-child)', // I allow all columns for dragdrop except the last
            'dataSrc' => 'sortsequence',
            'update' => false // this is key to prevent DT auto update
          ]
        ]
      )
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])

      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->orderBy(1)

      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_agent"),
          'className' => 'btn btn-primary ms-2',

          'attr' => [
            'title' =>  __("agents.add_agent"),
            'onclick' => "Livewire.dispatch('openAddUserModal')",


          ],
        ])

      ]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [

      Column::make('username')->title(__("users.username")),
      Column::make('name')->title(__("users.name")),
      Column::make('role')->title(__("users.role")),
      Column::make('company')->title(__("users.agent")),

      Column::make('phone')->title(__("users.phone")),

      Column::computed('status')->title(__("users.status"))
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),
      Column::computed('2fa')->title(__("users.2fa"))
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),
      Column::computed('action')->title(__("actions.actions"))
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),



    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'User_' . date('YmdHis');
  }
}
