<?php

namespace App\DataTables\Agents;

use App\Models\Agent;
use App\Models\BookingTransferTable;
use App\Models\Currency;
use App\Models\LedgerBooking;
use App\Models\LedgerCashier;
use App\Models\LedgerTransfer;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use PhpOffice\PhpSpreadsheet\Calculation\Financial\Securities\Price;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class BookingDataTable extends DataTable
{
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->setRowClass(function ($query) {
        return $query->status === 'completed' ? 'table-success' : ($query->status === 'pending' ? 'table-warning' : 'table-danger');
      })
      ->addColumn('action', function ($data) {
        return view('Booking.actions', compact('data'));
      })
      ->addColumn("from", function ($query) {
        return $query->sender->name ?? 'N/A';
      })
      ->addColumn("to", function ($query) {
        return $query->receiver->name ?? 'N/A';
      })
      ->addColumn("debit_entry", function ($query) {
        $html = "";
        $amount = $query->amount;
        $currency = $query->SenderCurrency->name ?? 'N/A';
        $color = $query->SenderCurrency->color ?? 'N/A';
        $fee = $query->send_fee;
        $html .= "<span>" . PriceFormat($amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:blue'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })

      ->addColumn("credit_entry", function ($query) {
        $html = "";
        $amount = $query->delivery_amount;
        $currency = $query->DeliveredCurrency->name ?? 'N/A';
        $color = $query->DeliveredCurrency->color ?? 'N/A';
        $fee = $query->receiver_fee ?? 0;
        $html .= "<span>" . PriceFormat($query->delivery_amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:red'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })
      ->addColumn("amount", function ($query) {
        return PriceFormat($query->amount);
      })
      ->editColumn("status", function ($query) {
        return "<span class='badge bg-" . $query->status_color . "'>" . __($query->status) . "</span>";
      })
      ->addColumn("created_by", function ($query) {
        return $query->user->name ?? 'N/A';
      })
      ->addColumn("created_at", function ($query) {
        return $query->created_at->format('Y-m-d H:i');
      })
      ->editColumn("note", function ($query) {
        return $query->note ?? __("general.no_note");;
      })
      ->addColumn("profit", function ($query) {
        return PriceFormat($query->send_fee - $query->receive_fee);
      })
      ->addColumn("customer", function ($query) {
        return  $query->recipient_name ?? 'N/A';
      })
      ->rawColumns(['action', 'debit_entry', 'credit_entry', 'amount', 'status', 'created_at'])
      ->setRowId('id');
  }

  public function query(LedgerBooking $model): QueryBuilder
  {
    $agent = auth()->guard('agent')->user()->agent->rootAgent()->id;

    $childs = Agent::where('parent_id', $agent)->pluck('id');
    $query = $model->newQuery()->withTrashed()
      ->with([
        'debitJournalEntry.details',
        'creditJournalEntry.details',
        'sender',
        'receiver',
        'DeliveredCurrency',
        'SenderCurrency',
        'user'
      ])
      ->whereIn('receiver_id', $childs)->orwhereIn('sender_id', $childs)
      ->orderBy('created_at', 'desc');



    return $query;
  }

  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'processing' => true,
        'serverSide' => true,
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])
      ->setTableId('transfer-table')
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',

      ])
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->orderBy(7, 'desc') // Order by created_at descending

      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_agent"),
          'className' => 'btn btn-primary ms-2',
          'action' => 'function (e, dt, node, config) { window.location = "/Booking/add"; }',
          'attr' => [
            'title' =>  __("agents.add_agent"),

          ],
        ]),
        Button::raw([
          'text' => '<i class="ti ti-search"></i> ',
          'className' => 'btn btn-info ms-2',
          'attr' => [
            'title' =>  __("general.search"),
            'data-bs-toggle' => 'offcanvas',
            'data-bs-target' => '#FilterModal',
          ],
        ])
      ]);
  }

  public function getColumns(): array
  {
    return [

      Column::make("created_at")
        ->title(__('transfers.created_at'))
        ->width('10%'),
      Column::make("debit_entry")
        ->title(__('transfers.debit'))
        ->width('10%')
        ->searchable(false),
      Column::make("from")
        ->title(__('transfers.from'))
        ->width('12%'),
      Column::make("customer")->title(__('transfers.receiver_client'))->width('12%'),
      Column::make("credit_entry")
        ->title(__('transfers.credit'))
        ->width('10%')
        ->searchable(false),
      Column::make("to")
        ->title(__('transfers.to'))
        ->width('12%'),
      Column::make("profit")
        ->title(__('transfers.profit'))
        ->width('5%'),
      Column::make("notes")
        ->title(__('transfers.notes'))
        ->width('10%'),
      Column::make("created_by")
        ->title(__('transfers.created_by'))
        ->width('10%'),
      Column::make("reference")
        ->title(__('transfers.reference'))
        ->width('20%'),

      Column::make("status")
        ->title(__('transfers.status'))
        ->width('20%'),







      Column::computed('action')
        ->title(__('Actions'))
        ->exportable(false)
        ->printable(false)
        ->width('6%')
        ->addClass('text-center'),
    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'BookingTransferTable_' . date('YmdHis');
  }
}
