<?php

namespace App\DataTables\Agents;

use App\Models\LedgerTransfer;
use App\Models\OutTransfer;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class InTransfersDataTable extends DataTable
{
    protected $filters = [];
    public function  __construct($filters = [])
    {
        $this->filters = $filters;
    }
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            // ->setRowClass(function ($query) {
            //     return $query->status === 'completed' ? 'table-success' : ($query->status === 'pending' ? 'table-warning' : 'table-danger');
            // })
            ->addColumn('action', function ($data) {
                $type = "in";
                return view('AgentPanel.Transfer.actions', compact('data', 'type'));
            })
            ->addColumn("from", function ($query) {
                return $query->sender->name ?? 'N/A';
            })
            ->addColumn("to", function ($query) {
                return $query->receiver->name ?? 'N/A';
            })


            ->addColumn("amount", function ($query) {
                return PriceFormat($query->amount, $query->currency);
            })
            ->editColumn("status", function ($query) {
                return "<span class='badge bg-" . $query->status_color . "'>" . __($query->status) . "</span>";
            })
            ->addColumn("created_by", function ($query) {
                return $query->user->name ?? 'N/A';
            })
            ->editColumn("created_at", function ($query) {
                return $query->created_at->format('Y-m-d H:i');
            })
            ->editColumn("delivered_at", function ($query) {
                if ($query->delivered_at == null) return 'N/A';
                return $query->delivered_at->format('Y-m-d H:i');
            })

            ->addColumn("profit", function ($query) {
                return PriceFormat($query->send_fee, $query->currency);
            })
            ->addColumn("details", function ($query) {
                $html = getTransferDetails($query);
                return $html;
            })
            ->editColumn("note", function ($query) {
                return $query->note ?? __("general.no_note");;
            })

            ->addColumn("customer", function ($query) {
                return  $query->recipient_name ?? 'N/A';
            })
            ->rawColumns(['action', 'debit_entry', 'credit_entry', 'amount', 'status', 'created_at', 'profit', 'note', 'customer', 'details'])
            ->setRowId('id');
    }

    public function query(LedgerTransfer $model): QueryBuilder
    {

        $agent =     auth()->guard('agent')->user()->agent->rootAgent();

        $query = $model->newQuery()->Transfer()->where('receiver_id', $agent->id)
            ->with([

                'sender',
                'receiver',
                'DeliveredCurrency',
                'SenderCurrency',

            ])
            ->orderBy('created_at', 'desc');
        $filters = $this->filters;

        if (request()->filled('filters')) {


            $filters = request()->get('filters');
        }

        $query = $query->Filter($filters);


        return $query;
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->parameters([
                'language' => [
                    'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
                ],
                'processing' => true,
                'serverSide' => true,
                'stateSave' => true,
                'deferRender' => true,
                'scrollX' => true,
                'pageLength' => 10, // Reduce default page length
                'lengthMenu' => [10, 25, 50, 100],
                'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
            ])
            ->select(false)
            ->setTableId('transfer-table')
            ->columns($this->getColumns())
            ->ajax([
                'url' => '', // current URL
                'type' => 'GET',
                'data' => 'function(data) {
                 const urlParams = new URLSearchParams(window.location.search);
    const show = urlParams.get("show");
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
        if (show) {
        data.filters.show = show;
    };
        data.search_term = $("#general-search").val(); // input field for general search

    }'
            ])
            ->dom('<"row"' .
                '<"col-md-2"<"ms-n2"l>>' .
                '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
                '>r' . // <-- add this line
                't' .
                '<"row"' .
                '<"col-sm-12 col-md-6"i>' .
                '<"col-sm-12 col-md-6"p>' .
                '>')



            ->buttons([

                Button::raw([
                    'text' => '<i class="ti ti-search"></i> ',
                    'className' => 'btn btn-info ms-2',
                    'attr' => [
                        'title' =>  __("general.search"),
                        'data-bs-toggle' => 'offcanvas',
                        'data-bs-target' => '#FilterModal',
                    ],
                ])
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::make("reference")
                ->title(__('transfers.reference'))
                ->width('20%'),
            Column::make("details")->title(__('transfers.receiver_client'))->width('20%'),
            Column::make("amount")
                ->title(__('transfers.amount'))
                ->width('10%'),
            Column::make("profit")
                ->title(__('transfers.profit'))
                ->width('5%'),
            Column::make("created_at")
                ->title(__('transfers.created_at'))
                ->width('10%'),
            Column::make("delivered_at")
                ->title(__('transfers.delivered_at'))
                ->width('10%'),




            Column::make("status")
                ->title(__('transfers.status'))
                ->width('20%'),








            Column::computed('action')
                ->title(__('Actions'))
                ->exportable(false)
                ->printable(false)
                ->width('6%')
                ->addClass('text-center'),
        ];
    }
    protected function filename(): string
    {
        return 'OutTransfers_' . date('YmdHis');
    }
}
