<?php

namespace App\DataTables;

use App\Models\Agent;
use App\Models\Branch;
use CountryCodes;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use PeterColes\Countries\CountriesFacade;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class BranchDataTable extends DataTable
{
  protected $filters;
  public function __construct($filters = [])
  {
    $this->filters = $filters;
  }
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('action', function ($data) {
        return view('Agents.branch_actions', compact('data'));
      })
      ->addColumn('agent', function ($query) {
        if (!$query->parent) {
          return '<span class="text-muted">No Parent Agent</span>';
        }
        return $query->parent->name ?? '';
      })
      ->addColumn('country', function ($query) {
        if (!$query->address) {
          return '<span class="fw-bold">🌍 No Address</span>';
        }

        $name = $query->address->country ? CountriesFacade::countryName($query->address->country, app()->getLocale()) : '';
        $icon = CountryCodes::getEmojiByAlpha2($query->address->country ?? 'SY');
        return '<span class="fw-bold">' . $icon . ' ' . $name . '</span>';
      })
      ->addColumn('state', function ($query) {
        return $query->address->state ?? '';
      })
      ->addColumn('city', function ($query) {
        return $query->address->city ?? '';
      })
      ->addColumn('neighborhood', function ($query) {
        return $query->address->neighborhood ?? '';
      })
      ->editColumn('created_at', function ($query) {
        return $query->created_at ? $query->created_at->format('Y-m-d H:i:s') : '';
      })
      ->editColumn('updated_at', function ($query) {
        return $query->updated_at ? $query->updated_at->format('Y-m-d
    H:i:s') : '';
      })
      ->filterColumn("agent", function ($query, $keyword) {
        $query->whereHas('parent', function ($q) use ($keyword) {
          $q->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn("country", function ($query, $keyword) {
        // Use helper function to get country codes for both Arabic and English search
        $matchedCodes = getCountryCodesForFilter($keyword);

        if (!empty($matchedCodes)) {
          $query->whereHas('address', function ($q) use ($matchedCodes) {
            $q->whereIn('country', $matchedCodes);
          });
        } else {
          // Fallback to original behavior if no match found
          $query->whereHas('address', function ($q) use ($keyword) {
            $q->where('country', 'like', "%{$keyword}%");
          });
        }
      })


      ->filterColumn("city", function ($query, $keyword) {
        $query->whereHas('address', function ($q) use ($keyword) {
          $q->where('city', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('type', function ($query, $keyword) {
        // Use helper function to get matched types for both Arabic and English
        $matchedTypes = getAgentTypesForFilter($keyword);

        if (!empty($matchedTypes)) {
          $query->whereIn('type', $matchedTypes);
        } else {
          // Fallback to original behavior if no match found
          $query->where('type', 'like', "%{$keyword}%");
        }
      })

      ->rawColumns(['action', 'country', 'agent'])
      ->setRowId('id');
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(Agent $model): QueryBuilder
  {
    $query = $model::SubAgents()->newQuery()->with("address", 'parent');
    if (isset($this->filters['parent_id'])) {
      $query->where('parent_id', $this->filters['parent_id']);
    }
    return $query;
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('branch-table')
      ->parameters(
        [
          'language' => [
            'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
          ],
          'processing' => true,
          'serverSide' => true,
        ]
      )
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])

      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      //->dom('Bfrtip')
      ->orderBy(1)

      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_branch"),
          'className' => 'btn btn-primary ms-2',
          'action' => 'function (e, dt, node, config) {Livewire.dispatch("openAddBranchModal", {"action_type":"add","agent_type":"branch"}); }',

        ])
      ]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [
      Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),
      Column::make('id'),
      Column::make('name')
        ->title(__('general.name')),
      Column::make('agent_id')
        ->title(__('agents.parent_agent'))
        ->data('agent')
        ->name('agent'),
      Column::computed('country')
        ->title(__('address.country'))
        ->searchable(true)
        ->orderable(true),

      Column::computed('city')
        ->title(__('address.city'))
        ->searchable(true)
        ->orderable(true),



    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'Branch_' . date('YmdHis');
  }
}
