<?php

namespace App\DataTables;

use Abivia\Ledger\Models\LedgerAccount;
use App\Models\CustomLedgerAccount;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class CurrencyPositionsDataTable extends DataTable
{
  protected array $currencies = [];

  public function __construct()
  {
    $this->currencies = \App\Models\Currency::pluck('code')->toArray();
  }

  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    $dataTable = new EloquentDataTable($query);

    // Account Code
    $dataTable->addColumn('account_code', fn($account) => $account->code);

    // Name (from LedgerAccount names relation)
    $dataTable->addColumn('name', function ($account) {
      return $account->names->first()->name ?? '';
    });

    // Currency Amount
    $dataTable->addColumn('currency_amount', function ($account) {
      $extra = is_string($account->extra) ? json_decode($account->extra, true) : $account->extra;
      $currencyCode = $extra['currency_code'] ?? null;

      if (!$currencyCode) return '<span class="text-danger">0.00</span>';

      $balance = $account->balances->firstWhere('currency', $currencyCode);
      $formatted = $balance ? PriceFormat(round(-1 * $balance->balance, 0)) : '0.00';
      $color = ($balance && -1 * $balance->balance < 0) ? 'text-red-600' : 'text-blue-600';
      return "<span class='{$color}'>{$formatted}</span>";
    })->escapeColumns([]); // escapeColumns([]) allows HTML rendering

    // Base Amount (USD)
    $dataTable->addColumn('base_amount', function ($account) {
      $balance = $account->balances->firstWhere('currency', "USD");
      $formatted = $balance ? PriceFormat(round(-1 * $balance->balance, 0)) : '0.00';
      $color = ($balance && -1 * $balance->balance < 0) ? 'text-red-600' : 'text-blue-600';
      return "<span class='{$color}'>{$formatted}</span>";
    })->escapeColumns([]);

    // Cost
    $dataTable->addColumn('cost', function ($account) {
      $extra = is_string($account->extra) ? json_decode($account->extra, true) : $account->extra;
      $currencyCode = $extra['currency_code'] ?? null;
      if (!$currencyCode) return '<span class="text-red-600">0.00</span>';
      Log::info("Currency code: {$currencyCode}");
      $cost = getExchangeCostPerUnitForTable("USD", $currencyCode);
      $color = ($cost > 0) ? 'text-blue-600' : 'text-red-600';
      return "<span class='{$color}'>{$cost}</span>";
    })->escapeColumns([]);

    return $dataTable;
  }

  protected function getCurrencyBalance($account, $currency)
  {
    $balance = $account->balances->firstWhere('currency', $currency);
    return $balance ? number_format($balance->balance, 2) : '0.00';
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(CustomLedgerAccount $model): QueryBuilder
  {
    return $model->newQuery()

      ->where("category", 0)
      ->whereNotNull("extra->currency_code")
      ->with("balances", "names");
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('currencypositions-table')
      ->columns($this->getColumns())
      ->minifiedAjax()
      ->orderBy(1)
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'processing' => true,
        'serverSide' => true,
        'deferRender' => true, // Improves performance for large datasets
        'scrollX' => true,
        'pageLength' => 25, // Optimize page length
        'lengthMenu' => [10, 25, 50, 100],
        'headerCallback' => 'function(thead, data, start, end, display) {
                    $(thead).css("background-color", "#685dd8");
                    $(thead).find("th").css({
                        "color": "white",
                        "font-weight": "bold",
                        "text-align": "center"
                    });
                }',
        'createdRow' => 'function(row, data, dataIndex) {
                    $(row).find("td").css("text-align", "center");
                }',
        'stripeClasses' => ['odd-row', 'even-row']
      ])
      ->buttons([]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return  $base = [
      Column::make('account_code')
        ->title(__('Account Code'))
        ->addClass('text-center'),
      Column::make('name')
        ->title(__('Name'))
        ->addClass('text-center'),
      Column::make('currency_amount')
        ->title(__('Currency'))
        ->addClass('text-center'),
      Column::make('base_amount')
        ->title(__('Balance'))
        ->addClass('text-center'),
      Column::make('cost')
        ->title(__('Cost'))
        ->addClass('text-center'),

    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'CurrencyPositions_' . date('YmdHis');
  }
}
