<?php

namespace App\DataTables;

use App\Models\Currency;
use App\Models\CurrencyTable;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class CurrencyTableDataTable extends DataTable
{
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('action', function ($data) {
        return view('Currency.actions', compact('data'));
      })
      ->editColumn("name", function ($query) {
        $color = $query->color;
        return '<span class="fw-bold">' . '<span style="color:' . $color . '">' . $query->name . '</span>';
      })
      ->editColumn("rate_factor", function ($query) {
        return __($query->rate_factor);
      })
      ->rawColumns(['action', 'name'])
      ->setRowId('id')

      ->setRowClass(function ($currency) {
        return $currency->is_default ? 'table-success' : '';
      });
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(Currency $model): QueryBuilder
  {
    return $model->newQuery()->orderBy("order_id", "asc");
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('currencytable-table')
      ->parameters(
        [
          'language' => [
            'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
          ],
          'processing' => true,
          'serverSide' => true,
          'rowReorder' => [
            'selector' => 'tr>td:not(:last-child)', // I allow all columns for dragdrop except the last
            'dataSrc' => 'sortsequence',
            'update' => false // this is key to prevent DT auto update
          ],
          'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
        ]
      )
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',

      ])

      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->orderBy(1)

      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_agent"),
          'className' => 'btn btn-primary ms-2',
          'action' => 'function (e, dt, node, config) {Livewire.dispatch("openAddCurrencyModal"); }',
        ])

      ]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [


      Column::make('name')->title(__("currency.name")),
      Column::make('code')->title(__("currency.code")),


      Column::make('exchange_rate')->title(__("currency.exchange_rate")),
      Column::make('rate_factor')->title(__("currency.rate_factor")),
      Column::make('order_id')->title(__("currency.order_id"))->width("10%"),
      Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),


    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'CurrencyTable_' . date('YmdHis');
  }
}
