<?php

namespace App\DataTables;

use App\Models\Agent;
use CountryCodes;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use PeterColes\Countries\CountriesFacade;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class DeletedAgentsDataTable extends DataTable
{
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('action',  function ($data) {
        return view('Agents.actions', compact('data'));
      })
      ->addColumn("country", function ($query) {
        $name = CountriesFacade::countryName($query->address->country ?? 'SY', app()->getLocale());
        $icon = CountryCodes::getEmojiByAlpha2($query->address->country ?? 'SY');
        return '<span class="fw-bold">' . $icon . ' ' . $name . '</span>';
      })


      ->editColumn("type", function ($query) {
        return $query->getTypeHtml();
      })

      ->addColumn("city", function ($query) {
        return $query->address->city ?? '';
      })
      ->filterColumn("country", function ($query, $keyword) {
        // Use helper function to get country codes for both Arabic and English search
        $matchedCodes = getCountryCodesForFilter($keyword);

        if (!empty($matchedCodes)) {
          $query->whereHas('address', function ($q) use ($matchedCodes) {
            $q->whereIn('country', $matchedCodes);
          });
        } else {
          // Fallback to original behavior if no match found
          $query->whereHas('address', function ($q) use ($keyword) {
            $q->where('country', 'like', "%{$keyword}%");
          });
        }
      })
      ->filterColumn('type', function ($query, $keyword) {
        // Use helper function to get matched types for both Arabic and English
        $matchedTypes = getAgentTypesForFilter($keyword);

        if (!empty($matchedTypes)) {
          $query->whereIn('type', $matchedTypes);
        } else {
          // Fallback to original behavior if no match found
          $query->where('type', 'like', "%{$keyword}%");
        }
      })
      ->rawColumns(['action', 'balances', 'total_balance', 'country', 'type'])

      ->setRowId('id');
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(Agent $model): QueryBuilder
  {
    return $model->newQuery()->onlyTrashed()->whereIn("type", ["agent", "account"])->whereNull('parent_id')->with("address");
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('agent-table')
      ->parameters(
        [
          'language' => [
            'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
          ],
          'processing' => true,
          'serverSide' => true,
          'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
        ]
      )
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',

      ])

      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      //->dom('Bfrtip')
      ->orderBy(0, 'desc')

      ->buttons([]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [


      Column::make('name')->title(__("agents.name")),
      Column::make('code')->title(__("agents.code")),
      Column::make('type')->title(__("agents.type")),
      Column::make('phone')->title(__("agents.phone")),
      Column::make("country")->title(__("agents.country")),
      Column::make("city")->title(__("agents.city")),

      Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),
    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'Agent_' . date('YmdHis');
  }
}
