<?php

namespace App\DataTables;

use App\Models\DeletedBooking;
use App\Models\LedgerBooking;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class DeletedBookingsDataTable extends DataTable
{
  public $filters;
  public $limit;
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */

  public function __construct($filters = [], $limit = null)
  {
    $this->filters = $filters;
    $this->limit = $limit;
  }
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      // ->setRowClass(function ($query) {
      //   return in_array($query->status, ['completed', 'booking'])
      //     ? 'table-success'
      //     : ($query->status === 'pending'
      //       ? 'table-warning'
      //       : 'table-danger');
      // })
      ->addColumn('action', function ($data) {
        return view('Booking.actions', compact('data'));
      })
      ->addColumn("from", function ($query) {
        return getAgentHtml($query->sender) ?? 'N/A';
      })
      ->addColumn("to", function ($query) {
        return getAgentHtml($query->receiver) ?? 'N/A';
      })
      ->addColumn("debit_entry", function ($query) {
        $html = "";
        $amount = $query->amount;
        $currency = $query->SenderCurrency->name ?? 'N/A';
        $color = $query->SenderCurrency->color ?? 'N/A';
        $fee = $query->send_fee;
        $html .= "<span>" . PriceFormat($amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:blue'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })
      ->addColumn("credit_entry", function ($query) {
        $html = "";
        $amount = $query->delivery_amount;
        $currency = $query->DeliveredCurrency->name ?? 'N/A';
        $color = $query->DeliveredCurrency->color ?? 'N/A';
        $fee = $query->receiver_fee ?? 0;
        $html .= "<span>" . PriceFormat($query->delivery_amount) . "</span><br>";
        $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
        $html .= "<span style='color:red'>(" . PriceFormat($fee) . ")</span>";
        return $html;
      })
      ->addColumn("amount", function ($query) {
        return PriceFormat($query->amount);
      })
      ->editColumn("status", function ($query) {
        return "<span class='badge bg-" . $query->status_color . "'>" . __($query->status) . "</span>";
      })
      ->addColumn("created_by", function ($query) {
        return $query->user->name ?? 'N/A';
      })
      ->addColumn("created_at", function ($query) {
        return $query->created_at->format('Y-m-d H:i');
      })
      ->editColumn("notes", function ($query) {
        $user = $query->user->name ?? "";
        return $query->notes . " - " . '<i class="fas fa-user me-1"></i>' . $user;
      })






      ->addColumn("customer", function ($query) {
        return  $query->recipient_name ?? $query->customer_details['name'] ?? 'N/A';
      })
      /////////////////////////Filters/////////////////////////
      ->filterColumn('from', function ($query, $keyword) {
        $query->whereHas('sender', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('to', function ($query, $keyword) {
        $query->whereHas('receiver', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('customer', function ($query, $keyword) {
        $query->where('recipient_name', 'like', "%{$keyword}%");
      })
      ->filterColumn('note', function ($query, $keyword) {
        $query->where('notes', 'like', "%{$keyword}%");
      })
      ->filterColumn('debit_entry', function ($query, $keyword) {
        $query->where('amount', 'like', "%{$keyword}%");
      })
      ->filterColumn('credit_entry', function ($query, $keyword) {
        $query->where('delivery_amount', 'like', "%{$keyword}%");
      })
      ->filterColumn('status', function ($query, $keyword) {
        $query->where('status', 'like', "%{$keyword}%");
      })
      ->filterColumn('created_at', function ($query, $keyword) {
        $query->where('created_at', 'like', "%{$keyword}%");
      })
      ->filterColumn('created_by', function ($query, $keyword) {
        $query->whereHas('user', function ($query) use ($keyword) {
          $query->where('name', 'like', "%{$keyword}%");
        });
      })
      ->filterColumn('reference', function ($query, $keyword) {
        $query->where('reference', 'like', "%{$keyword}%");
      })
      /////////////////////////Ordering/////////////////////////
      ->orderColumn('reference', function ($query, $order) {
        $query->orderBy('reference', $order);
      })
      ->orderColumn('from', function ($query, $order) {
        $query->orderBy('sender_id', $order);
      })
      ->orderColumn('to', function ($query, $order) {
        $query->orderBy('receiver_id', $order);
      })
      ->orderColumn('customer', function ($query, $order) {
        $query->orderBy('recipient_name', $order);
      })
      ->orderColumn('debit_entry', function ($query, $order) {
        $query->orderBy('amount', $order);
      })
      ->orderColumn('credit_entry', function ($query, $order) {
        $query->orderBy('delivery_amount', $order);
      })
      ->orderColumn('profit', function ($query, $order) {
        $query->orderByRaw("(send_fee - receiver_fee) {$order}");
      })
      ->orderColumn('created_by', function ($query, $order) {
        $query->orderBy('user_id', $order);
      })
      ->orderColumn('status', function ($query, $order) {
        $query->orderBy('status', $order);
      })
      ->orderColumn('notes', function ($query, $order) {
        $query->orderBy('notes', $order);
      })
      ->orderColumn('created_at', function ($query, $order) {
        $query->orderBy('created_at', $order);
      })
      ->rawColumns(['action', 'debit_entry', 'credit_entry', 'amount', 'status', 'created_at', 'from', 'to', 'customer', 'notes'])
      ->setRowId('id');
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(LedgerBooking $model): QueryBuilder
  {
    $query = $model->newQuery()
      ->with([
        'debitJournalEntry.details',
        'creditJournalEntry.details',
        'sender',
        'receiver',
        'DeliveredCurrency',
        'SenderCurrency',
        'user'
      ])
      ->orderBy('created_at', 'desc');

    // Apply filters first
    if (request()->filled('filters')) {
      $filters = request()->get('filters');
      $query = $model->Filter($filters);
    } else {
      $query = $model->Filter($this->filters);
    }

    // Apply limit if set
    if ($this->limit) {
      $query->limit($this->limit);
    }

    // Apply onlyTrashed AFTER filters to ensure it's not overridden
    $query->onlyTrashed();

    // Eager load relationships AFTER applying filters to avoid N+1 queries
    $query->with([
      'debitJournalEntry.details',
      'creditJournalEntry.details',
      'sender' => function ($query) {
        $query->select('id', 'name', 'code', 'type', 'is_active');
      },
      'receiver' => function ($query) {
        $query->select('id', 'name', 'code', 'type', 'is_active');
      },
      'DeliveredCurrency' => function ($query) {
        $query->select('code', 'name', 'color', 'order_id');
      },
      'SenderCurrency' => function ($query) {
        $query->select('code', 'name', 'color', 'order_id');
      },
      'user' => function ($query) {
        $query->select('id', 'name');
      }
    ]);

    return $query;
  }
  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'processing' => true,
        'serverSide' => true,
        'searching' => true,
        'ordering' => true,
        'pageLength' => 25,
        'lengthMenu' => [[10, 25, 50, 100, -1], [10, 25, 50, 100, 'All']],
        'search' => [
          'smart' => true,
          'regex' => false,
          'caseInsensitive' => true,
        ],
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])
      ->setTableId('booking-table')
      ->columns($this->getColumns())
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }',
      ])
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>t' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->orderBy(1, 'desc') // Order by created_at descending (column index 1)
      ->buttons([
        Button::raw([
          'text' => '<i class="ti ti-plus"></i> ' . __("agents.add_agent"),
          'className' => 'btn btn-primary ms-2',
          'action' => 'function (e, dt, node, config) { window.location = "/Booking/add"; }',
          'attr' => [
            'title' =>  __("agents.add_agent"),
            'id' => 'add-booking',

          ],
        ]),
        Button::raw([
          'text' => '<i class="ti ti-search"></i> ',
          'className' => 'btn btn-info ms-2',
          'attr' => [
            'id' => 'search-booking',
            'title' =>  __("general.search"),
            'data-bs-toggle' => 'offcanvas',
            'data-bs-target' => '#FilterModal',
          ],
        ])
      ]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [
      Column::make("reference")
        ->title(__('transfers.reference'))
        ->width('20%')
        ->searchable(true)
        ->orderable(true),

      Column::make("created_at")
        ->title(__('transfers.created_at'))
        ->width('10%')
        ->searchable(true)
        ->orderable(true),

      Column::make("debit_entry")
        ->title(__('transfers.debit'))
        ->width('10%')
        ->searchable(false) // HTML column, ordering handled via ->orderColumn()
        ->orderable(true),

      Column::make("from")
        ->title(__('transfers.from'))
        ->width('12%')
        ->searchable(true)
        ->orderable(true),

      Column::make("customer")
        ->title(__('transfers.receiver_client'))
        ->width('10%')
        ->searchable(true)
        ->orderable(true),

      Column::make("credit_entry")
        ->title(__('transfers.credit'))
        ->width('9%')
        ->searchable(false) // same reason as debit_entry
        ->orderable(true),

      Column::make("to")
        ->title(__('transfers.to'))
        ->width('10%')
        ->searchable(true)
        ->orderable(true),



      Column::make("notes") // ✅ FIXED (was "notes")
        ->title(__('transfers.notes'))
        ->width('20%')
        ->searchable(true)
        ->orderable(true),



      Column::make("status")
        ->title(__('transfers.status'))
        ->width('20%')
        ->searchable(true)
        ->orderable(true),

      Column::computed('action')
        ->title(__('Actions'))
        ->exportable(false)
        ->printable(false)
        ->width('6%')
        ->addClass('text-center')
        ->searchable(false)
        ->orderable(false),
    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'DeletedBookings_' . date('YmdHis');
  }
}
