<?php

namespace App\DataTables;

use Abivia\Ledger\Models\JournalDetail;
use App\Models\Journal;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Abivia\Ledger\Models\JournalEntry;
use App\Models\Agent;
use App\Models\JournalDetailOverwrite;

class JournalDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('debit', function ($detail) {
                return $detail->amount > 0 ? format_money($detail->amount, $detail->entry->currency) : '';
            })
            ->addColumn('credit', function ($detail) {
                return $detail->amount < 0 ? format_money(abs($detail->amount), $detail->entry->currency) : '';
            })
            ->addColumn('transaction_type', function ($detail) {
                // Adjust if you have a real type, else static label
                return 'حركة';
            })
            ->addColumn('account_number', function ($detail) {
                return $detail->account?->names->where("language", app()->getLocale())->first()->name . "(" . $detail->account?->code . ")";
            })

            ->addColumn('description', function ($detail) {
                return $detail->entry?->description ?? '-';
            })
            ->addColumn('entry_date', function ($detail) {
                return $detail->entry?->transDate   ?? '-';
            })
            ->addColumn('document_number', function ($detail) {
                // Replace with your actual document/ref field if exists
                return $detail->entry->transfer->reference ?? '-';
            })

            ->addColumn('user', function ($detail) {
                // Assuming createdBy or updatedBy user name available in entry or detail
                // Replace this with your actual user relation or field
                return $detail->entry?->transfer->user->name ?? '-';
            })
            ->addColumn('entry_id', function ($detail) {
                return $detail->journalEntryId;
            })
            ->addColumn('detail_id', function ($detail) {
                return $detail->journalDetailId;
            })
            ->filterColumn("document_number", function ($query, $keyword) {
                $query->whereHas('entry.transfer', function ($query) use ($keyword) {
                    $query->where('reference', 'like', '%' . $keyword . '%');
                });
            })
            ->filterColumn("account_number", function ($query, $keyword) {
                $query->whereHas('account', function ($query) use ($keyword) {
                    $query->where('code', 'like', '%' . $keyword . '%');
                })->orwhereHas('account.names', function ($query) use ($keyword) {
                    $query->where('name', 'like', '%' . $keyword . '%');
                });
            })
            ->rawColumns(['debit', 'credit', 'transaction_type', 'account_number', 'branch', 'description', 'entry_date', 'document_number', 'user', 'entry_id', 'detail_id'])
            ->setRowId('journalDetailId');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(JournalDetailOverwrite $model): QueryBuilder
    {
        return $model->newQuery()->with('entry', 'entry.transfer', 'account', 'entry.transfer.user');
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('journal-table')
            ->columns($this->getColumns())
            ->minifiedAjax()

            //->dom('Bfrtip')
            ->orderBy(1)
            ->parameters([
                'language' => [
                    'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
                ],
                'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
                'pageLength' => 1000, // Reduce default page length
                'lengthMenu' => [100, 500, 1000, 2000],
            ])
            ->buttons([]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('credit')->title('لنا'),
            Column::make('debit')->title('علينا'),

            Column::make('transaction_type')->title('نوع الحركة'),
            Column::make('account_number')->title('رقم الحساب'),

            Column::make('description')->title('البيان'),
            Column::make('entry_date')->title('تاريخ القيد'),
            Column::make('document_number')->title('رقم المستند'),
            Column::make('user')->title('المستخدم'),
            Column::make('entry_id')->title('رقم القيد'),
            Column::make('detail_id')->title('رقم الحركة'),

        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Journal_' . date('YmdHis');
    }
}
