<?php

namespace App\DataTables;

use App\Models\Agent;
use App\Models\Currency;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class SypRatesDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return datatables()
            ->eloquent($query)
            ->addColumn('syp_rates', function (Agent $agent) {

                return view('Agents.partials.syp_rates', [
                    'agent' => $agent,
                    'currencies' => Currency::where('code', '!=', 'SYP')->get(), // all non-SYP currencies
                ])->render();
            })
            ->rawColumns(['syp_rates']);
    }

    public function query(Agent $model): QueryBuilder
    {
        return $model->newQuery()->select('id', 'name');
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('syp-rates-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->parameters([
                'processing' => true,
                'serverSide' => true,
                'pageLength' => 50,
                'searchDelay' => 500,
                'language' => [
                    'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
                ],
                'headerCallback' => 'function(thead) {
                    $(thead).css("background-color", "#685dd8");
                    $(thead).find("th").css({"color": "white", "font-weight": "bold"});
                }',
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::make('id')->title('ID'),
            Column::make('name')->title('الاسم'),

            Column::computed('syp_rates')
                ->title('اسعار التحويل بالليرة السورية')
                ->exportable(false)
                ->printable(false)
                ->addClass('text-left'),
        ];
    }

    protected function filename(): string
    {
        return 'SypRates_' . date('YmdHis');
    }
}
