<?php

use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\LedgerBalance;
use App\Enums\CurrencyEnum;
use App\Enums\WalletSlug;
use App\Exceptions\InsufficientCurrencyBalance;
use App\Exceptions\InvalidExchangeRateException;
use App\Models\Agent;
use App\Models\ApiConfig;
use App\Models\Currency;
use App\Models\Evlat;
use App\Models\Exchange;
use App\Models\LedgerAccountOverwrite;
use App\Models\TransferPackage;
use App\Models\User;
use App\Services\ExchangeService;
use App\Services\NotificationSender;
use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Str;
use NumberToWords\NumberToWords;
use Ramsey\Uuid\Type\Decimal;

use App\Interfaces\PartnerApiInterface;
use App\Models\LedgerTransfer;
use Rawilk\Settings\Facades\Settings;

function PriceFormat($price, $currency = null)
{

  $lang = App::getLocale();
  $color = null;
  if ($currency) {
    $currency = Currency::getCurrencyName($currency);
    $color = Currency::getCurrencyColor($currency);
  }
  // Check if number is integer (or has zero decimals)
  if (floor($price) == $price) {
    // integer, no decimals
    $formattedPrice = number_format($price, 0, '.', ','); // change separators if needed
  } else {
    // has decimals, show 2 decimal places
    $formattedPrice = number_format($price, 2, '.', ',');
  }
  if ($currency == null) return $formattedPrice;
  return $formattedPrice . ' ' . "<span style='color: {$color}'>{$currency}</span>";
}
function NoStylePriceFormat($price, $currency = null)
{

  $lang = App::getLocale();
  $color = null;
  if ($currency) {
    $currency = Currency::getCurrencyName($currency);
    $color = Currency::getCurrencyColor($currency);
  }
  // Check if number is integer (or has zero decimals)
  if (floor($price) == $price) {
    // integer, no decimals
    $formattedPrice = number_format($price, 0, '.', ','); // change separators if needed
  } else {
    // has decimals, show 2 decimal places
    $formattedPrice = number_format($price, 2, '.', ',');
  }
  if ($currency == null) return $formattedPrice;
  return $formattedPrice . ' ' . "{$currency}";
}
if (!function_exists('format_money')) {
  function format_money($price, $currency = null)
  {
    static $currencyCache = [];

    $lang = App::getLocale();
    $color = null;

    if ($currency) {
      if (! isset($currencyCache[$currency])) {
        $currencyCache[$currency] = [
          'name' => Currency::getCurrencyName($currency),
          'color' => Currency::getCurrencyColor($currency),
        ];
      }

      $currencyName = $currencyCache[$currency]['name'];
      $color = $currencyCache[$currency]['color'];
    }

    // number formatting
    $formattedPrice = floor($price) == $price
      ? number_format($price, 0, '.', ',')
      : number_format($price, 2, '.', ',');

    if (! $currency) {
      return $formattedPrice;
    }

    return $formattedPrice . ' ' . "<span style='color: {$color}'>{$currencyName}</span>";
  }
}
if (!function_exists('number_format')) {
  function number_format($price)
  {



    if (floor($price) == $price) {
      // integer, no decimals
      $formattedPrice = number_format($price, 0, '.', ','); // change separators if needed
    } else {
      // has decimals, show 2 decimal places
      $formattedPrice = number_format($price, 2, '.', ',');
    }

    return $formattedPrice;
  }
}


function areaFormat($space)
{
  $formatted_space = number_format($space, 2, '.', ',');
  return  $formatted_space . " m²";
}
function generateAvatar($name)
{
  $nameParts = explode(' ', $name);
  $initials = strtoupper(substr($nameParts[0], 0, 1) . (isset($nameParts[1]) ? substr($nameParts[1], 0, 1) : ''));

  $image = imagecreatetruecolor(100, 100);
  $backgroundColor = imagecolorallocate($image, 0, 123, 255); // Blue background
  $textColor = imagecolorallocate($image, 255, 255, 255); // White text

  imagefilledrectangle($image, 0, 0, 100, 100, $backgroundColor);
  $fontPath = public_path('fonts/OpenSans-Bold.ttf'); // Ensure the font exists
  imagettftext($image, 30, 0, 20, 60, $textColor, $fontPath, $initials);

  $avatarPath = 'avatars/' . uniqid() . '.png';
  imagepng($image, public_path($avatarPath));
  imagedestroy($image);

  return asset($avatarPath);
}
function generateSvgAvatar($name)
{
  $nameParts = explode(' ', trim($name));
  $initials = strtoupper(substr($nameParts[0], 0, 1) . (isset($nameParts[1]) ? substr($nameParts[1], 0, 1) : ''));

  $bgColor = '#007bff'; // Bootstrap blue
  $textColor = '#ffffff';

  $svg = <<<SVG
<svg xmlns="http://www.w3.org/2000/svg" width="100" height="100" viewBox="0 0 100 100">
    <circle cx="50" cy="50" r="48" fill="$bgColor" />
    <text x="50%" y="54%" dominant-baseline="middle" text-anchor="middle"
          font-size="36" fill="$textColor" font-family="Segoe UI, Arial, sans-serif" font-weight="bold">
        $initials
    </text>
</svg>
SVG;

  return 'data:image/svg+xml;base64,' . base64_encode($svg);
}





function generateMainCariCode($id, $type = "tenant"): string
{

  $type = Str::lower($type);
  $prefix = match ($type) {
    'property' => 'PROP',
    'tenant'   => 'TEN',
    'contract' => 'Lease',
    default    => 'GEN',
  };
  return 'MLK-' . str_pad($id, 7, '0', STR_PAD_LEFT);
}

/**
 * Generate a structured sub-cari code for a given main cari code.
 *
 * @param string $mainCode The main cari code
 * @param int $index The index of the sub-cari code
 * @return string The generated sub-cari code
 */
function generateSubCariCode($mainCode, $index): string
{
  return $mainCode . '.' . str_pad($index, 3, '0', STR_PAD_LEFT);
}

function generateNestedCariCode(string $baseCode, array $levels): string
{
  foreach ($levels as $level) {
    $baseCode .= '.' . str_pad($level, 2, '0', STR_PAD_LEFT);
  }
  return $baseCode;
}
function generateCariCode(string $prefix, int $tenantId,   $contractId = null, array $segments = []): string
{
  $code = $prefix . '-' . str_pad($tenantId, 7, '0', STR_PAD_LEFT);
  $code .= '.' . str_pad($contractId, 3, '0', STR_PAD_LEFT);

  foreach ($segments as $segment) {
    $code .= '.' . str_pad($segment, 2, '0', STR_PAD_LEFT);
  }

  return $code;
}
function getCariCode(int $tenantId, int $contractId, array $segments = []): string
{
  $prefix = "MLK";
  return $prefix . '-' . str_pad($tenantId, 7, '0', STR_PAD_LEFT) . '.' . str_pad($contractId, 3, '0', STR_PAD_LEFT);
}
if (!function_exists('generateWalletSlug')) {
  /**
   * Generate a structured wallet slug for tenants, properties, or contracts.
   *
   * @param string $entityType 'tenant' or 'property'
   * @param int $entityId The ID of the tenant or property
   * @param int|null $contractNumber Optional contract number if used
   * @param string $walletType Wallet type slug like 'rent', 'deposit', 'maintenance'
   * @return string
   */
  function generateWalletSlug(string $entityType, int $entityId, ?int $contractNumber, string $walletType): string
  {
    $entityPrefix = match ($entityType) {
      'tenant'   => WalletSlug::Tenant,
      'property' => WalletSlug::Property,
      default    => throw new InvalidArgumentException('Invalid entity type'),
    };

    $entityCode = $entityPrefix . '-' . str_pad($entityId, 7, '0', STR_PAD_LEFT);
    $walletCode = match ($walletType) {
      'rent'        => '01',
      'deposit'     => '02',
      'maintenance' => '03',
      default       => '99', // Unknown type
    };

    if ($contractNumber !== null) {
      $contractCode = str_pad($contractNumber, 2, '0', STR_PAD_LEFT);
      return "{$entityCode}.{$contractCode}.{$walletCode}";
    }

    return "{$entityCode}.{$walletCode}";
  }
}

if (!function_exists('walletBalanceColor')) {
  function walletBalanceColor(float $balance): string
  {
    return $balance >= 0 ? 'text-success' : 'text-danger';
  }
}
function extractTenantInfoFromDescription($description)
{
  $info = [];

  // Match national ID (11 digits)
  if (preg_match('/\b(\d{11})\b/', $description, $matches)) {
    $info['national_id'] = $matches[1];
  }

  // Match invoice code like INV-202405
  if (preg_match('/INV-\d{6}/i', $description, $matches)) {
    $info['invoice_code'] = $matches[0];
  }

  // Match flat/daire number like "Flat 12A" or "Daire 3"
  if (preg_match('/(?:Flat|Daire)\s?(\w+)/i', $description, $matches)) {
    $info['flat'] = $matches[1];
  }

  // Match simple names (e.g., Firstname Lastname)
  if (preg_match('/([A-ZÇĞİÖŞÜ][a-zçğıöşü]+)\s+([A-ZÇĞİÖŞÜ][a-zçğıöşü]+)/u', $description, $matches)) {
    $info['name'] = $matches[0]; // Full name
  }

  // Match custom code like KOD1234
  if (preg_match('/KOD\d+/i', $description, $matches)) {
    $info['code'] = $matches[0];
  }

  return $info;
}
function normalizeName($name)
{
  return strtolower(str_replace(['ç', 'ğ', 'ı', 'ö', 'ş', 'ü'], ['c', 'g', 'i', 'o', 's', 'u'], $name));
}
function matchTenant($description, $tenantList)
{
  $chunks = extractNameChunks($description);
  $bestMatch = null;
  $bestScore = 0;

  foreach ($chunks as $chunk) {
    foreach ($tenantList as $tenant) {
      $descChunk = normalizeName($chunk);
      $tenantName = normalizeName($tenant->name);
      similar_text($descChunk, $tenantName, $percent);
      if ($percent > $bestScore) {
        $bestScore = $percent;
        $bestMatch = $tenant;
      }
    }
  }

  return $bestScore > 70 ? $bestMatch : null;
}

function extractNameChunks($description)
{
  // Normalize and clean
  $text = normalizeName($description);

  // Tokenize
  $words = preg_split('/\s+/', $text);

  // Keep only word groups that look like names (2+ alphabetic words)
  $chunks = [];
  for ($i = 0; $i < count($words) - 1; $i++) {
    if (ctype_alpha($words[$i]) && ctype_alpha($words[$i + 1])) {
      $chunks[] = $words[$i] . ' ' . $words[$i + 1];
    }
    if ($i < count($words) - 2 && ctype_alpha($words[$i]) && ctype_alpha($words[$i + 1]) && ctype_alpha($words[$i + 2])) {
      $chunks[] = $words[$i] . ' ' . $words[$i + 1] . ' ' . $words[$i + 2];
    }
  }
  return array_unique($chunks);
}

function extractMonth($description)
{
  $months = [
    'ocak' => 1,
    'subat' => 2,
    'mart' => 3,
    'nisan' => 4,
    'mayis' => 5,
    'haziran' => 6,
    'temmuz' => 7,
    'agustos' => 8,
    'eylul' => 9,
    'ekim' => 10,
    'kasim' => 11,
    'aralik' => 12
  ];

  $description = strtolower($description);
  foreach ($months as $name => $num) {
    if (strpos($description, $name) !== false) {
      // Try to extract year too
      preg_match('/\d{4}/', $description, $matches);
      $year = $matches[0] ?? now()->year;
      return sprintf('%04d-%02d', $year, $num);
    }
  }

  return null; // Month not found
}
function autoMapColumns(array $headers)
{
  $targets = [
    'date' => ['tarih', 'date', 'islem tarihi'],
    'amount' => ['tutar', 'amount', 'miktar'],
    'description' => ['açıklama', 'description', 'işlem açıklaması'],
    'name' => ['gönderen', 'gonderen', 'ad soyad', 'name'],
  ];

  $map = [];

  foreach ($targets as $key => $keywords) {
    foreach ($headers as $i => $header) {
      $normalized = strtolower(trim($header));
      if (in_array($normalized, $keywords)) {
        $map[$key] = $i;
        break;
      }
    }
  }

  return $map;
}
function FormatBalanceColor($balance)
{
  return $balance < 0 ? 'text-danger' : 'text-success';
}
if (!function_exists('calculateTax')) {
  /**
   * Calculate tax based on mode.
   *
   * @param float $amount Original amount
   * @param float $taxRate Tax rate in percentage (e.g., 18 for 18%)
   * @param string $type 'included' or 'excluded'
   * @return array ['net' => ..., 'tax' => ..., 'total' => ...]
   */
  function calculateTax(float $amount, float $taxRate, bool $included = false): array
  {
    $taxRateDecimal = $taxRate / 100;

    if ($included) {
      $net = $amount / (1 + $taxRateDecimal);
      $tax = $amount - $net;
      $total = $amount;
    } else { // 'excluded'
      $tax = $amount * $taxRateDecimal;
      $net = $amount;
      $total = $net + $tax;
    }

    return [
      'net' => round($net, 2),
      'tax' => round($tax, 2),
      'total' => round($total, 2),
    ];
  }
}
if (! function_exists('divnum')) {

  function divnum($numerator, $denominator)
  {
    return $denominator == 0 ? 0 : ($numerator / $denominator);
  }
}
if (!function_exists('calculateLateInterest')) {
  function calculateLateInterest($rent, $overdue)
  {
    if (!$overdue['overdue_rate'] || !$overdue['overdue_start']) return 0;

    $today = Carbon::today();
    $due_date = Carbon::createFromDate($rent['rent_start'])->day($rent['payment_day']);
    if ($today->lessThan($due_date->addDays($overdue['overdue_start']))) {
      return 0; // not yet overdue
    }

    $daysLate = $today->diffInDays($due_date);
    $rate = floatval($overdue['overdue_rate']);

    // Example: Simple daily interest
    return ($daysLate * ($rate / 100) * $rent['rent_amount']) / 30; // Approx monthly rate
  }
}
if (!function_exists('normalize_gender')) {
  function normalize_gender($input): ?string
  {
    $value = strtolower(trim($input));

    $maleInputs = ['erkek', 'e', 'male', 'm'];
    $femaleInputs = ['kadın', 'kadin', 'k', 'female', 'f', 'kız', 'kiz'];

    if (in_array($value, $maleInputs)) {
      return 'm';
    }

    if (in_array($value, $femaleInputs)) {
      return 'f';
    }

    return null; // unknown/invalid
  }
}
function resolve_name_columns(array $data): array
{
  $first = trim($data['firstname'] ?? '');
  $last = trim($data['lastname'] ?? '');

  // If both are filled and NOT the same, assume they are correctly mapped
  if ($first && $last && $first !== $last) {
    return [
      'firstname' => $first,
      'lastname' => $last,
    ];
  }

  // Otherwise, try to split from a single full name
  $full = $first ?: ($last ?: ($data['name'] ?? $data['fullname'] ?? ''));

  $parts = preg_split('/\s+/', trim($full));
  $lastname = array_pop($parts);
  $firstname = implode(' ', $parts);

  return [
    'firstname' => $firstname,
    'lastname' => $lastname,
  ];
}


function generate_unique_email(string $firstname, string $lastname): string
{
  $base = Str::slug($firstname . '.' . $lastname, '.');
  $domain = 'mulktakibi.com';

  $email = $base . '@' . $domain;
  $counter = 1;

  // Ensure uniqueness
  while (User::where('email', $email)->exists()) {
    $email = $base . $counter++ . '@' . $domain;
  }

  return $email;
}
function  GetActionTitle($action)
{
  switch ($action) {
    case "deposit_odeme":
      return __("general.deposit_odeme");
    case "rent_tahsilati":
      return __("general.rent_tahsilati");
    case "deposit_tahsilati":
      return __("general.deposit_tahsilati");
    case "rent_tahsilati":
      return __("general.rent_tahsilati");
    case "policy_tahsilati":
      return __("general.policy_tahsilati");
    case "policy_odeme":
      return __("general.policy_odeme");
    case "payment":
      return __("general.payment");
    default:
      return "Collection";
  }
}
function buildAccountTree($accounts, $parentUuid = null)
{
  $branch = [];

  foreach ($accounts as $account) {
    if ($account->parentUuid === $parentUuid) {
      $children = buildAccountTree($accounts, $account->ledgerUuid);
      $node = [
        'account' => $account,
        'children' => $children,
      ];
      $branch[] = $node;
    }
  }
  return $branch;
}
function getLedgerAccountName(LedgerAccount $account, string $lang = 'en'): ?string
{
  return $account->names
    ->where('language', $lang)
    ->pluck('name')
    ->first();
}
function toWords(int|float $number): string
{
  $lang = App::getLocale();
  $numberToWords = new NumberToWords();

  try {
    $numberTransformer = $numberToWords->getNumberTransformer($lang);
    return $numberTransformer->toWords($number);
  } catch (\Throwable $e) {
    return (string) $number; // fallback
  }
}
function toCurrencyWords(float $number,  string $currency = 'USD'): string
{
  $lang = App::getLocale();
  // العملة من CurrencyEnum حسب اللغة المطلوبة

  $currencyName = Currency::where('code', $currency)->first()->name;
  if ($number !== 0) {
    $words = toWords((int)$number);
    return "{$words} {$currencyName}";
  }


  return "{$currencyName}";
}
function CalculateRevenue($amount, $send_fee, $receive_fee, $send_type = "fixed", $receive_type = "fixed")
{
  if ($send_type === "fixed") {
    $send_fee = $send_fee;
  } else {
    $send_fee = $amount * $send_fee / 100;
  }
  if ($receive_type === "fixed") {
    $receive_fee = $receive_fee;
  } else {
    $receive_fee = $amount * $receive_fee / 100;
  }
  $send = is_numeric($send_fee) ? (float)$send_fee : 0;
  $receive = is_numeric($receive_fee) ? (float)$receive_fee : 0;
  return $send - $receive;
}
function  Exchange($from_amount, $rate, $factor, $type = "normal")
{
  // $factor can be "multiply" or "divide"
  // $type can be "normal" or "reverse"
  if ($type === "reverse") {
    $factor = $factor === "multiply" ? "divide" : "multiply";
  }

  if ($rate <= 0) {
    return 0; // Avoid division by zero
  }

  return calculateExchange($from_amount, $rate, $factor);
}
function calculateExchange($from_amount, $rate, $factor)
{
  if ($factor == "multiply") {
    return $from_amount * $rate;
  } else {
    return $from_amount / $rate;
  }
}
function convertCurrency($fromAmount, $fromCurrency, $toCurrency, $currencies)
{

  $default = collect($currencies)->firstWhere('is_default', 1);
  $defaultCode = $default['code'];

  if ($fromCurrency === $toCurrency) {
    return $fromAmount;
  }

  $fromData = collect($currencies)->firstWhere('code', $fromCurrency);
  $toData = collect($currencies)->firstWhere('code', $toCurrency);

  // Case A: Direct conversion
  if ($fromCurrency === $defaultCode) {
    // normal
    return Exchange($fromAmount, $toData['exchange_rate'], $toData['rate_factor'], 'normal');
  } elseif ($toCurrency === $defaultCode) {
    // reverse
    return Exchange($fromAmount, $fromData['exchange_rate'], $fromData['rate_factor'], 'reverse');
  }

  // Case B: Convert from → default → to
  $toDefault = Exchange($fromAmount, $fromData['exchange_rate'], $fromData['rate_factor'], 'reverse');
  return Exchange($toDefault, $toData['exchange_rate'], $toData['rate_factor'], 'normal');
}

function getBaseCurrencyBalance($balances)
{
  Log::info($balances);
  $baseCurrency = 'USD';
  $total = 0;
  $html = "علينا";
  foreach ($balances as $balance) {
    $amount = (float) ($balance->balance ?? $balance['balance'] ?? 0);

    if ($balance->currency ?? $balance['currency'] === $baseCurrency) {
      $total = $total + $amount;
    } else {
      $currency = Currency::where("code", $balance->currency ?? $balance['currency'])->first();
      if ($currency && $currency->exchange_rate && $currency->rate_factor) {
        $baseBalance = Exchange($amount, $currency->exchange_rate, $currency->rate_factor, "normal");

        Log::info("amount" . $amount . " currency: " .   $balance['currency'] . " rate: " . $currency->exchange_rate . " factor: " . $currency->rate_factor . "Base balance: " . $baseBalance);
        $total = $total + $baseBalance;
      }
    }
  }
  Log::info("Total: " . $total);
  if ($total < 0) {
    $html = "لنا";
  }

  return  round($total, 4); // Optional: Round to 2 decimals
}


/**
 * Get agent types for filtering based on search keyword (supports both Arabic and English)
 */
function getAgentTypesForFilter($keyword)
{
  // Create reverse mapping from Arabic to English types
  $typeMapping = [
    'مركز' => 'agent',
    'موظف' => 'cashier',
    'عميل' => 'account',
    'عابر' => 'branch',
    // English fallbacks
    'agent' => 'agent',
    'cashier' => 'cashier',
    'account' => 'account',
    'branch' => 'branch'
  ];

  // Check if the keyword matches any Arabic or English type
  $matchedTypes = [];
  foreach ($typeMapping as $searchTerm => $type) {
    if (stripos($searchTerm, $keyword) !== false || stripos($keyword, $searchTerm) !== false) {
      $matchedTypes[] = $type;
    }
  }

  return $matchedTypes;
}

function getFeePriceFromPackage($package_id, $amount)
{
  $package = TransferPackage::with("details")->find($package_id);

  if ($package) {

    $details = $package->details->where("min_amount", "<=", $amount)->where("max_amount", ">=", $amount)->first();
    if (!$details) {
      return [
        'status' => false,
        'send_fee' => 0,
        'receive_fee' => 0,
        'msg' => __('No fee details found for the specified amount.')
      ];
    }
    $send_fee = $details->send_fee ?? 0;
    $receive_fee = $details->receive_fee ?? 0;

    if ($details->send_fee_type == "percent") {
      $send_fee = $amount * $details->send_fee / 1000;
    }
    if ($details->receive_fee_type == "percent") {
      $receive_fee = $amount * $details->receive_fee / 1000;
    }
    return [
      'status' => true,
      'send_fee' => $send_fee,
      'receive_fee' => $receive_fee
    ];
  }
  return [
    'status' => false,
    'send_fee' => 0,
    'receive_fee' => 0,
    'msg' => 'No fee details found for the specified amount.'
  ];
}
function convertAmount($amount, $fromCurrency, $toCurrency)
{
  if ($fromCurrency === $toCurrency) {
    return $amount;
  }

  $rate = \App\Models\ExchangeRate::where('from_currency', $fromCurrency)
    ->where('to_currency', $toCurrency)
    ->latest('rate_date')
    ->first();

  if ($rate) {
    return round($amount * $rate->rate, 2);
  }

  // Try reversed
  $reverseRate = \App\Models\ExchangeRate::where('from_currency', $toCurrency)
    ->where('to_currency', $fromCurrency)
    ->latest('rate_date')
    ->first();

  if ($reverseRate) {
    return round($amount / $reverseRate->rate, 2);
  }

  throw new \Exception("Exchange rate not found in either direction.");
}
function generateAgentAccountCode(string $baseCode, int $agentId): string
{
  $agent = Agent::findOrFail($agentId);

  if ($agent->type === 'cashier') {
    $parent = $agent->parent;
    $agentSuffix = str_pad($parent->id, 7, '0', STR_PAD_LEFT); // e.g., 01
    $cashierSuffix = str_pad($agent->id, 7, '0', STR_PAD_LEFT); // e.g., 03
    $prefix = substr($baseCode, 0, 4); // e.g., "11"

    return $prefix . $agentSuffix . $cashierSuffix; // e.g., 110103
  }

  $agentSuffix = str_pad($agentId, 7, '0', STR_PAD_LEFT); // e.g., 01
  $prefix = substr($baseCode, 0, 2); // e.g., "11"

  return $prefix . $agentSuffix; // e.g., 1101
}
function generateCurrencyAccountCode(string $baseCode, int $currencyId)
{
  // baseCode: like "1400" for exchange root
  // Reserve a block of 10 per currency: 1410, 1420, 1430...
  $code = (int)$baseCode + ($currencyId * 10);
  return (string)$code;
}

function generateAgentCurrencyCashCode(int $agentId, int $currencyId): string
{
  $cashCode = generateAgentAccountCode('1100', $agentId); // e.g., 1101
  return $cashCode . str_pad($currencyId, 7, '0', STR_PAD_LEFT); // e.g., 110101
}
function getAgentCurrencyCashCode(int $agentId, string $prefix, string $currencyCode): string
{
  Log::info("Getting agent currency cash code for agent ID: {$agentId}, prefix: {$prefix}, currency code: {$currencyCode}");
  $account = LedgerAccount::where('extra->agent_id', $agentId)->where('extra->currency_code', $currencyCode)
    ->where('code', 'like', $prefix . '%') // Assuming cash accounts start with '11'
    ->first();

  return (string)$account->code;
}
// ... existing code ...

/**
 * Generate a reference number based on configured settings
 *
 * @param string|null $prefix Optional custom prefix (overrides settings)
 * @return string Generated reference number
 * @throws \InvalidArgumentException When settings are invalid
 */
function generateReferenceNumber(?string $prefix = null): string
{
  // Get settings from cache with fallback to defaults
  $settings = getReferenceNumberSettings();

  // Validate settings
  validateReferenceNumberSettings($settings);

  // Use provided prefix or default from settings
  $finalPrefix = $prefix ?? $settings['prefix'];

  // Generate components based on settings
  $branchCode = generateBranchCode($settings['branch_code_length']);
  $transactionId = generateTransactionId($settings, $settings['transaction_id_length']);

  // Build reference using custom format
  return buildReferenceFromFormat($finalPrefix, $branchCode, $transactionId, $settings);
}

/**
 * Get reference number settings from cache or return defaults
 *
 * @return array<string, mixed> Settings array
 */
function getReferenceNumberSettings(): array
{
  $defaultSettings = [
    'prefix' => 'REF',
    'branch_code_length' => 3,
    'transaction_id_length' => 6,
    'separator' => '-',
    'auto_increment' => false,
    'reset_daily' => false,
    'custom_format' => '{prefix}{separator}{branch_code}{separator}{transaction_id}'
  ];

  $cachedSettings = \Illuminate\Support\Facades\Cache::get('reference_number_settings');

  if ($cachedSettings && is_array($cachedSettings)) {
    return array_merge($defaultSettings, $cachedSettings);
  }

  return $defaultSettings;
}

/**
 * Validate reference number settings
 *
 * @param array<string, mixed> $settings Settings to validate
 * @throws \InvalidArgumentException When settings are invalid
 */
function validateReferenceNumberSettings(array $settings): void
{

  $requiredKeys = ['prefix', 'branch_code_length', 'transaction_id_length', 'separator', 'custom_format'];

  foreach ($requiredKeys as $key) {
    if (!isset($settings[$key])) {
      throw new \InvalidArgumentException("Missing required setting: {$key}");
    }
  }
  $settings['branch_code_length'] = (int)$settings['branch_code_length'];
  $settings['transaction_id_length'] = (int)$settings['transaction_id_length'];

  if (!is_string($settings['prefix']) || strlen($settings['prefix']) > 10) {
    throw new \InvalidArgumentException('Prefix must be a string with maximum length of 10');
  }

  if (!is_int($settings['branch_code_length']) || $settings['branch_code_length'] < 1 || $settings['branch_code_length'] > 10) {
    throw new \InvalidArgumentException('Branch code length must be an integer between 1 and 10');
  }

  if (!is_int($settings['transaction_id_length']) || $settings['transaction_id_length'] < 1 || $settings['transaction_id_length'] > 10) {
    throw new \InvalidArgumentException('Transaction ID length must be an integer between 1 and 10');
  }

  if (!is_string($settings['separator']) || strlen($settings['separator']) > 5) {
    throw new \InvalidArgumentException('Separator must be a string with maximum length of 5');
  }

  if (!is_string($settings['custom_format'])) {
    throw new \InvalidArgumentException('Custom format must be a string');
  }
}

/**
 * Generate branch code with specified length
 *
 * @param int $length Length of branch code
 * @return string Generated branch code
 */
function generateBranchCode(int $length): string
{
  $maxValue = pow(10, $length) - 1;
  $randomValue = mt_rand(0, $maxValue);

  return str_pad((string) $randomValue, $length, '0', STR_PAD_LEFT);
}

/**
 * Generate transaction ID based on settings
 *
 * @param array<string, mixed> $settings Reference number settings
 * @param int $length Length of transaction ID
 * @return string Generated transaction ID
 */
function generateTransactionId(array $settings, int $length): string
{
  if (isset($settings['auto_increment']) && $settings['auto_increment'] === true) {
    $sequence = getNextSequence($settings);
    return str_pad((string) $sequence, $length, '0', STR_PAD_LEFT);
  }

  // Fallback to random generation
  $maxValue = pow(10, $length) - 1;
  $randomValue = mt_rand(0, $maxValue);

  return str_pad((string) $randomValue, $length, '0', STR_PAD_LEFT);
}

/**
 * Get next sequence number for auto-increment
 *
 * @param array<string, mixed> $settings Reference number settings
 * @return int Next sequence number
 */
function getNextSequence(array $settings): int
{
  if (isset($settings['reset_daily']) && $settings['reset_daily'] === true) {
    $key = 'reference_sequence_' . date('Y-m-d');
    return (int) \Illuminate\Support\Facades\Cache::increment($key);
  }

  // For non-daily reset, use a persistent sequence
  $key = 'reference_sequence_persistent';

  // Get current value or initialize to 0
  $currentValue = \Illuminate\Support\Facades\Cache::get($key, 0);

  // Increment and store
  $newValue = $currentValue + 1;
  \Illuminate\Support\Facades\Cache::put($key, $newValue, now()->addYear());

  return $newValue;
}

/**
 * Build reference number from custom format
 *
 * @param string $prefix Reference prefix
 * @param string $branchCode Generated branch code
 * @param string $transactionId Generated transaction ID
 * @param array<string, mixed> $settings Reference number settings
 * @return string Formatted reference number
 */
function buildReferenceFromFormat(string $prefix, string $branchCode, string $transactionId, array $settings): string
{
  $format = $settings['custom_format'];
  $separator = $settings['separator'];

  $placeholders = [
    '{prefix}' => $prefix,
    '{separator}' => $separator,
    '{branch_code}' => $branchCode,
    '{transaction_id}' => $transactionId
  ];

  return str_replace(array_keys($placeholders), array_values($placeholders), $format);
}


function getExchangeCostPerUnit($fromCurrency, $toCurrency): float
{
  $baseCurrency = config('app.base_currency', 'USD');
  $from = Currency::where('code', strtoupper($fromCurrency))->first();
  $to = Currency::where('code', strtoupper($toCurrency))->first();
  $fromCurrency = strtoupper($fromCurrency);
  $toCurrency = strtoupper($toCurrency);

  if ($fromCurrency === $baseCurrency) {
    // Selling base currency, cost basis is in 'buy' account of the currency received (to_currency)
    $currency = Currency::where('code', $toCurrency)->firstOrFail();
    $buyAccountCode = $currency->GetExchangeAccountCode()['buy'] ?? $currency->GetExchangeAccountCode()['exchange'];


    $ledgerAccount = LedgerAccount::where('code', $buyAccountCode)->with('balances')->first();
    if (! $ledgerAccount) {
      throw new \Exception("Buy ledger account not found for currency {$toCurrency}");
    }

    $balances = $ledgerAccount->balances;

    $baseBalance = abs($balances->firstWhere('currency', $baseCurrency)?->balance ?? 0);
    if ($baseBalance == 0) {
      throw new InvalidExchangeRateException($fromCurrency, $toCurrency);
    }
    $currencyBalance = abs($balances->firstWhere('currency', $toCurrency)?->balance ?? 0);
    $cost =  ($baseBalance / $currencyBalance);

    if ($currencyBalance == 0) {
      return 0;
    }


    // Cost is how much base currency (USD) you spent per unit of other currency
    return  divnum($currencyBalance, $baseBalance);
  }

  // Else buying base currency, cost is not calculated or 0
  return 0;
}
function getExchangeCostPerUnitForTable($fromCurrency, $toCurrency): float
{

  $baseCurrency = config('app.base_currency', 'USD');
  $from = Currency::where('code', strtoupper($fromCurrency))->first();
  $to = Currency::where('code', strtoupper($toCurrency))->first();
  $fromCurrency = strtoupper($fromCurrency);
  $toCurrency = strtoupper($toCurrency);

  if ($fromCurrency === $baseCurrency) {
    // Selling base currency, cost basis is in 'buy' account of the currency received (to_currency)
    $currency = Currency::where('code', $toCurrency)->firstOrFail();
    $buyAccountCode = $currency->GetExchangeAccountCode()['buy'] ?? $currency->GetExchangeAccountCode()['exchange'];

    Log::info("Buy account code: {$buyAccountCode}");
    $ledgerAccount = LedgerAccount::where('code', $buyAccountCode)->with('balances')->first();
    if (! $ledgerAccount) {
      return 0; // throw new \Exception("Buy ledger account not found for currency {$toCurrency}");
    }
    Log::info("Ledger account: {$ledgerAccount}");
    $balances = $ledgerAccount->balances;

    $baseBalance = abs($balances->firstWhere('currency', $baseCurrency)?->balance ?? 0);
    if ($baseBalance == 0) {
      // throw new InvalidExchangeRateException($fromCurrency, $toCurrency);
    }
    $currencyBalance = abs($balances->firstWhere('currency', $toCurrency)?->balance ?? 0);
    Log::info("Base balance: {$baseBalance}, currency balance: {$currencyBalance} ");
    $cost =  divnum($baseBalance, $currencyBalance);


    if ($currencyBalance == 0) {
      return 0;
    }


    // Cost is how much base currency (USD) you spent per unit of other currency
    return  $currency->rate_factor == "divide" ? divnum($currencyBalance, $baseBalance) : divnum($baseBalance, $currencyBalance);
  }

  // Else buying base currency, cost is not calculated or 0
  return 0;
}
function getCurrencyCost(string $baseCurrencyCode): array
{
  $currency = Currency::where('code', $baseCurrencyCode)->first();
  if (!$currency) return [];

  $buyAccountCode = $currency->GetExchangeAccountCode()['buy'] ?? null;
  if (!$buyAccountCode) return [];

  $ledgerAccount = LedgerAccount::where('code', $buyAccountCode)->first();
  if (!$ledgerAccount) return [];

  $balances = LedgerBalance::where('ledgerUuid', $ledgerAccount->ledgerUuid)->get();

  $baseBalance = $balances->firstWhere('currency', $baseCurrencyCode)?->balance;
  $otherBalanceRecord = $balances->firstWhere(fn($b) => $b->currency !== $baseCurrencyCode);

  if (!$baseBalance || !$otherBalanceRecord || $baseBalance == 0 || $otherBalanceRecord->balance == 0) {
    return [];
  }

  return [
    'cost_per_unit' => abs($otherBalanceRecord->balance) / abs($baseBalance),
    'from_currency' => $otherBalanceRecord->currency,
    'to_currency' => $baseCurrencyCode,
  ];
}
function getCurrencyAvatar($currencyCode): string
{
  $currency = Currency::where('code', $currencyCode)->first();
  return $currency->getAvatar();
}
function getTemplateKeys($type = "transfer"): array
{
  if ($type == "transfer" || $type == "delivery" || $type == "booking") {
    return [
      'company_name' => __('general.company_name'),
      'logo' => __('general.logo'),
      'reference' => __('transfers.reference'),
      'sender_id' => __('transfers.sender_agent'),
      'receiver_id' => __('transfers.receiver_agent'),
      'sender_address' => __('transfers.sender_address'),
      'receiver_address' => __('transfers.receiver_address'),

      'created_at' =>  __('transfers.created_at'),
      'delivered_at' =>  __('transfers.delivered_at'),
      'client_name' =>  __('transfers.client_name'),
      'client_phone' =>  __('transfers.client_phone'),
      'secret' =>  __('transfers.secret'),
      'identity_number' =>  __('transfers.identity_number'),
      'notes' =>  __('transfers.notes'),
      'amount' =>  __('transfers.amount'),
      'delivery_amount' =>  __('transfers.delivery_amount'),
      'send_fee' =>  __('transfers.send_fee'),
      'receive_fee' =>  __('transfers.receive_fee'),


    ];
  } elseif ($type == "exchange") {
    return [
      'company_name' => __('general.company_name'),
      'logo' => __('general.logo'),
      'reference' => __('exchange.reference'),
      'agent_id' => __('exchange.agent'),
      'from_amount' => __('exchange.from_amount'),
      'from_currency' =>  __('exchange.from_currency'),
      'to_amount' =>  __('exchange.to_amount'),
      'to_currency' =>  __('exchange.to_currency'),
      'notes' =>  __('exchange.notes'),
      'date' =>  __('exchange.created_at'),
      'agent_address' =>  __('exchange.agent_address'),

    ];
  } else {
    return [];
  }
}
if (!function_exists('getAgentAccountCode')) {
  function getAgentAccountCode(int $agentId, string $prefix): string
  {
    $account = LedgerAccount::where('agent_id', $agentId)
      ->where('code', 'like', $prefix . '%')
      ->first();

    return $account->code ?? "";
  }
}
if (! function_exists('exchange_rate')) {
  function exchange_rate(string $from, string $to, string $type = 'balance'): ?float
  {
    if ($from == $to) return 1;
    try {
      if (isset($from) && isset($to))
        return app(\App\Services\ExchangeRates\ExchangePriceResolver::class)->getRate($from, $to, $type);
      return 1;
    } catch (\Exception $e) {
      throw new InvalidExchangeRateException($from, $to);
    }
  }
}
function ValidateCurrencyExchangeBalance(string $currencyCode, $amount)
{
  $currency = Currency::where('code', $currencyCode)->first();
  $balance = $currency->getCurrencyExchangeBalances();

  if ($balance < $amount) {
    throw new InsufficientCurrencyBalance($currencyCode, $amount);
  }
}
function getCurrencyFactor(?string $from, ?string $to, string $type = 'balance')
{
  try {
    if (!empty($from) && !empty($to)) {
      return app(\App\Services\ExchangeRates\ExchangePriceResolver::class)
        ->getFactor($from, $to, $type);
    }
    return "multiply";
  } catch (\Exception $e) {
    return "multiply";
  }
}

function getAPICompanies()
{
  return ApiConfig::pluck("id", "name")->toArray();
}


if (!function_exists('sendExternalTransaction')) {
  function sendExternalTransaction($config, array $transactionData, int $receiverId): array
  {

    $className = $config->apiType->class_name;
    // Instantiate the API service class
    /** @var PartnerApiInterface $apiService */
    $apiService = new $className();

    // Load and prepare config

    $configArray = json_decode($config->config_json, true);
    $configArray['base_url'] = $config->base_url;

    // Set API config
    $apiService->setConfig($configArray);

    // Merge and send transaction
    $transactionData['receiver_account_code'] = $config->agent_id;
    $transactionData['receiver_id'] = $receiverId;

    return $apiService->sendTransaction($transactionData);
  }
}
if (!function_exists('ApiConnection')) {
  function ApiConnection($config)
  {

    $className = $config->apiType->class_name;
    // Instantiate the API service class
    /** @var PartnerApiInterface $apiService */
    $apiService = new $className();

    // Load and prepare config

    $configArray = json_decode($config->config_json, true);
    $configArray['base_url'] = $config->base_url;

    // Set API config
    $apiService->setConfig($configArray);

    return $apiService;
  }
}
if (!function_exists('getBalanceReconciliationhtml')) {
  function getBalanceReconciliationhtml($html, $balances)
  {
    $text = "علينا";
    $based_balance = getBaseCurrencyBalance($balances);
    if ($based_balance < 0) {
      $text = "لنا";
    }

    $companyName = Settings::get('general.company_name');
    return str_replace([
      '%company_name%',
      '%date%',
      '%based_balance%'
    ], [
      $companyName, // Or get from DB
      now()->format('Y-m-d'),
      format_money($based_balance, "USD") . " $text"
    ], $html);
  }
}
if (!function_exists('generateRandomTransferNumber')) {
  function generateRandomTransferNumber($prefix = "TRF"): string
  {
    $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "{$prefix}-{$branchCode}-{$transactionId}";
  }
}
if (!function_exists('generateTransferDescription')) {
  function generateTransferDescription($transfer): string
  {
    $type = $transfer->type;
    $status = $transfer->status ?? "pending";
    if ($status == "reverse") {
      $data['old_reference'] = LedgerTransfer::withoutGlobalScopes()->find($transfer->extra['reverses_transfer_id'] ?? null)->reference ?? $transfer->extra['reverses_transfer_id'] ?? null;
    }
    $data = [
      'from_currency' => Currency::getCurrencyName($transfer->currency),
      'from_amount' => $transfer->amount,
      'to_currency' => Currency::getCurrencyName($transfer->delivery_currency),
      'to_amount' => $transfer->delivery_amount,
      'amount' => $transfer->amount,
      'from' => $transfer->sender->name ?? "",
      'to' => $transfer->receiver->name ?? "",
      'description' => $transfer->description,
      'rate' => $transfer->rate ?? 1,
      'reference' => $transfer->reference,
      'old_reference' => $transfer->extra['reverses_transfer_id'] ?? null,

    ];
    Log::info("Generating transfer description for type: $type, status: $status, data: " . json_encode($data));
    return trans("description.$type.$status", $data);
  }
}
if (!function_exists('getCurrencyShortcuts')) {
  function getCurrencyShortcuts()
  {
    return   Currency::getCurrencyShortcuts();
  }
}
if (!function_exists('getAgentHtml')) {
  function getAgentHtml($agent): string
  {
    if ($agent == null) {
      return "";
    }
    $uniqueId = 'agent_' . $agent->id . '_' . uniqid();
    $agent->parent_id != null ? $color = "text-dark" : $color = "text-dark";
    $agent->parent_id != null ? $name = $agent->name . " - " . __("Branch") : $name = $agent->name;
    return '<div class="text-center">
        <span class="fw-bold fs-5 d-block ' . $color . ' cursor-pointer "
              onclick="loadAgentBalances_' . $uniqueId . '(' . $agent->id . ', \'' . addslashes($agent->name) . '\')"
              style="cursor: pointer; ">
          ' . $name . '
        </span>
        <div class="dropdown d-inline-block">
          <div class="fw-bold badge ' . $color . ' fs-5 cursor-pointer"
               onclick="loadAgentBalances_' . $uniqueId . '(' . $agent->id . ', \'' . addslashes($agent->name) . '\')"
               style="background-color: #FFFF00; cursor: pointer;"
               role="button">
            ' . $agent->code . '
          </div>
          <div class="dropdown-menu dropdown-menu-center" style="padding: 20px; min-width: 300px;">
            <div class="text-center">
              <small class="text-muted">Click on agent name or code to view balances</small>
            </div>
          </div>
        </div>

        <!-- Custom popover for balances -->
        <div id="balance-popover-' . $agent->id . '" class="balance-popover" style="display: none; position: absolute; z-index: 9999; background: white; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); padding: 20px; min-width: 300px; max-width: 400px;">
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h6 class="mb-0">' . $agent->name . ' Balances</h6>
            <button type="button" class="btn-close" onclick="closeBalancePopover(' . $agent->id . ')" aria-label="Close"></button>
          </div>
          <div id="balance-content-' . $agent->id . '" class="text-center">
            <div class="spinner-border spinner-border-sm" role="status">
              <span class="visually-hidden">Loading...</span>
            </div>
            <small class="text-muted">Loading balances...</small>
          </div>
        </div>
      </div>
      <script>
      (function() {
        var uniqueId = "' . $uniqueId . '";
        if (typeof window["loadAgentBalances_" + uniqueId] === "undefined") {
          window["loadAgentBalances_" + uniqueId] = function(agentId, agentName) {
            console.log("Loading balances for agent:", agentId, agentName, "with unique ID:", uniqueId);

            // Close all other popovers first
            closeAllBalancePopovers();

            // Show the custom popover
            const popover = document.getElementById("balance-popover-" + agentId);
            const balanceContent = document.getElementById("balance-content-" + agentId);

            if (!popover || !balanceContent) {
              console.error("Popover elements not found for agent:", agentId);
              return;
            }

            // Position the popover closer to the agent name
            const agentNameElement = document.querySelector(\'[onclick*="loadAgentBalances_\' + uniqueId + \'"]\');
            if (agentNameElement) {
              const rect = agentNameElement.getBoundingClientRect();

              // Popover dimensions
              const popoverWidth = 350;
              const popoverHeight = 200;
              const viewportWidth = window.innerWidth;
              const viewportHeight = window.innerHeight;

              // Use viewport-relative positioning (no scroll offset needed)
              let left = rect.right + 5; // 5px gap to the right
              let top = rect.top + (rect.height / 2) - (popoverHeight / 2); // Perfect vertical centering

              // Check right edge - if popover goes off right side
              if (left + popoverWidth > viewportWidth - 20) {
                // Position to the left of the agent name
                left = rect.left - popoverWidth - 5;
              }

              // Check left edge - if popover goes off left side
              if (left < 20) {
                // If left side is also full, try positioning below the agent name
                if (rect.bottom + popoverHeight < viewportHeight - 20) {
                  left = rect.left + (rect.width / 2) - (popoverWidth / 2); // Center horizontally
                  top = rect.bottom + 5; // Below agent name
                } else {
                  // If below is also full, position above
                  left = rect.left + (rect.width / 2) - (popoverWidth / 2); // Center horizontally
                  top = rect.top - popoverHeight - 5; // Above agent name
                }
              }

              // Check bottom edge
              if (top + popoverHeight > viewportHeight - 20) {
                top = viewportHeight - popoverHeight - 20;
              }

              // Check top edge
              if (top < 20) {
                top = 20;
              }

              // Final boundary check to ensure popover is fully visible
              if (left < 20) left = 20;
              if (left + popoverWidth > viewportWidth - 20) left = viewportWidth - popoverWidth - 20;
              if (top < 20) top = 20;
              if (top + popoverHeight > viewportHeight - 20) top = viewportHeight - popoverHeight - 20;

              // Apply positioning using fixed positioning for better control
              popover.style.position = "fixed";
              popover.style.left = left + "px";
              popover.style.top = top + "px";

              console.log("Positioning popover:", {
                agentLeft: rect.left,
                agentRight: rect.right,
                agentWidth: rect.width,
                agentTop: rect.top,
                agentHeight: rect.height,
                finalLeft: left,
                finalTop: top,
                viewportWidth: viewportWidth,
                viewportHeight: viewportHeight,
                gap: "5px from agent name",
                positioning: left < rect.left ? "left of agent" : "right of agent"
              });
            }

            // Show popover
            popover.style.display = "block";

            // Show loading state
            balanceContent.innerHTML = \'<div class="spinner-border spinner-border-sm" role="status"><span class="visually-hidden">Loading...</span></div><br><small class="text-muted">Loading balances...</small>\';

            console.log("Making request to:", "/api/agent/" + agentId + "/balances");

            fetch("/api/agent/" + agentId + "/balances")
              .then(response => {
                console.log("Response status:", response.status);
                return response.json();
              })
              .then(data => {
                console.log("Response data:", data);
                if (data.success) {
                  console.log("Setting HTML content:", data.html);
                  balanceContent.innerHTML = data.html;

                  // Log the updated content
                  console.log("Updated balance content:", balanceContent.innerHTML);
                } else {
                  balanceContent.innerHTML = \'<div class="text-danger"><i class="fas fa-exclamation-triangle"></i> Error loading balances</div>\';
                }
              })
              .catch(error => {
                console.error("Error:", error);
                balanceContent.innerHTML = \'<div class="text-danger"><i class="fas fa-exclamation-triangle"></i> Network error</div>\';
              });
          };
        }
      })();

      // Global function to close popover
      if (typeof closeBalancePopover === "undefined") {
        function closeBalancePopover(agentId) {
          const popover = document.getElementById("balance-popover-" + agentId);
          if (popover) {
            popover.style.display = "none";
          }
        }
      }

      // Global function to close all popovers
      if (typeof closeAllBalancePopovers === "undefined") {
        function closeAllBalancePopovers() {
          const allPopovers = document.querySelectorAll(\'.balance-popover\');
          allPopovers.forEach(popover => {
            popover.style.display = "none";
          });
        }
      }
      </script>';
  }
}

if (!function_exists('getTransferDetails')) {
  function getTransferDetails($transfer): string
  {
    $sender = __("agent/transfers.sender") . ": <span class='fw-bold'>" . e($transfer->client?->name) . "</span>";
    $receiver = __("agent/transfers.receiver") . ": <span class='fw-bold'>" . e($transfer->receiver?->name) . "</span>";
    $recipient = __("agent/transfers.recipient") . ": <span class='fw-bold text-dark'>" . e($transfer->recipient_name) . "</span>";

    return '
      <div class="transfer-details">
          <div class="mb-1">' . $sender . '</div>
          <div class="mb-1">' . $receiver . '</div>
          <div>' . $recipient . '</div>
      </div>
  ';
  }
}

function getBalanceLimit($agentId)
{
  $agent = Agent::find($agentId);
  return $agent->GlobalCreditLimit->base_credit_limit ?? 0;
}

/**
 * Get country codes for filtering based on search keyword (supports both Arabic and English country names)
 */
function getCountryCodesForFilter($keyword)
{
  // Common countries with Arabic and English names mapped to country codes
  $countryMapping = [
    // Syria
    'سوريا' => 'SY',
    'سورية' => 'SY',
    'syria' => 'SY',
    'syrian' => 'SY',

    // Lebanon
    'لبنان' => 'LB',
    'lebanon' => 'LB',
    'lebanese' => 'LB',

    // Jordan
    'الأردن' => 'JO',
    'jordan' => 'JO',
    'jordanian' => 'JO',

    // Iraq
    'العراق' => 'IQ',
    'iraq' => 'IQ',
    'iraqi' => 'IQ',

    // Egypt
    'مصر' => 'EG',
    'egypt' => 'EG',
    'egyptian' => 'EG',

    // Saudi Arabia
    'السعودية' => 'SA',
    'السعوديه' => 'SA',
    'saudi arabia' => 'SA',
    'saudi' => 'SA',

    // UAE
    'الإمارات' => 'AE',
    'الإمارات العربية المتحدة' => 'AE',
    'united arab emirates' => 'AE',
    'uae' => 'AE',

    // Kuwait
    'الكويت' => 'KW',
    'kuwait' => 'KW',
    'kuwaiti' => 'KW',

    // Qatar
    'قطر' => 'QA',
    'qatar' => 'QA',
    'qatari' => 'QA',

    // Bahrain
    'البحرين' => 'BH',
    'bahrain' => 'BH',
    'bahraini' => 'BH',

    // Oman
    'عمان' => 'OM',
    'oman' => 'OM',
    'omani' => 'OM',

    // Yemen
    'اليمن' => 'YE',
    'yemen' => 'YE',
    'yemeni' => 'YE',

    // Turkey
    'تركيا' => 'TR',
    'turkey' => 'TR',
    'turkish' => 'TR',

    // Iran
    'إيران' => 'IR',
    'iran' => 'IR',
    'iranian' => 'IR',

    // Afghanistan
    'أفغانستان' => 'AF',
    'afghanistan' => 'AF',
    'afghan' => 'AF',

    // Pakistan
    'باكستان' => 'PK',
    'pakistan' => 'PK',
    'pakistani' => 'PK',

    // India
    'الهند' => 'IN',
    'india' => 'IN',
    'indian' => 'IN',

    // China
    'الصين' => 'CN',
    'china' => 'CN',
    'chinese' => 'CN',

    // Russia
    'روسيا' => 'RU',
    'russia' => 'RU',
    'russian' => 'RU',

    // United States
    'الولايات المتحدة' => 'US',
    'أمريكا' => 'US',
    'united states' => 'US',
    'usa' => 'US',
    'america' => 'US',

    // United Kingdom
    'المملكة المتحدة' => 'GB',
    'بريطانيا' => 'GB',
    'united kingdom' => 'GB',
    'uk' => 'GB',
    'britain' => 'GB',

    // France
    'فرنسا' => 'FR',
    'france' => 'FR',
    'french' => 'FR',

    // Germany
    'ألمانيا' => 'DE',
    'germany' => 'DE',
    'german' => 'DE',

    // Italy
    'إيطاليا' => 'IT',
    'italy' => 'IT',
    'italian' => 'IT',

    // Spain
    'إسبانيا' => 'ES',
    'spain' => 'ES',
    'spanish' => 'ES',

    // Canada
    'كندا' => 'CA',
    'canada' => 'CA',
    'canadian' => 'CA',

    // Australia
    'أستراليا' => 'AU',
    'australia' => 'AU',
    'australian' => 'AU',


  ];

  // Check if the keyword matches any country name
  $matchedCodes = [];
  foreach ($countryMapping as $searchTerm => $code) {
    if (stripos($searchTerm, $keyword) !== false || stripos($keyword, $searchTerm) !== false) {
      $matchedCodes[] = $code;
    }
  }

  return $matchedCodes;
}
function  getBalanceReconciliationBasedFormat($balance)
{
  $currency = Currency::where("is_default", 1)->first();
  $formated_balance = NoStylePriceFormat(abs($balance), $currency->code);
  $text = "لنا";
  $color = "blue";
  if ($balance > 0) {
    $text = "علينا";
    $color = "red";
  }
  $html = "<span class='text-{$color}'>{$formated_balance} {$text}</span> ";


  return $html;
}
function BlockTransfer($transfer)
{
  $transfer->is_blocked = true;
  $transfer->status = 'blocked';
  $transfer->save();
}
function UnblockTransfer($transfer)
{
  $transfer->is_blocked = false;
  $transfer->status = 'pending';
  $transfer->save();
}
