<?php

namespace App\Http\Controllers;

use App\Models\News;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NewsController extends Controller
{
  /**
   * Display a listing of the news.
   */
  public function index()
  {
    $news = News::orderBy('created_at', 'desc')->paginate(10);
    return view('news.index', compact('news'));
  }

  /**
   * Show the form for creating a new news.
   */
  public function create()
  {
    return view('news.create');
  }

  /**
   * Store a newly created news in storage.
   */
  public function store(Request $request)
  {
    $validator = Validator::make($request->all(), [
      'content' => 'required|string|min:10',
      'is_active' => 'boolean'
    ]);

    if ($validator->fails()) {
      return redirect()->back()
        ->withErrors($validator)
        ->withInput();
    }

    News::create([
      'content' => $request->content,
      'is_active' => $request->has('is_active')
    ]);

    return redirect()->route('news.index')
      ->with('success', 'تم إضافة الخبر بنجاح');
  }

  /**
   * Show the form for editing the specified news.
   */
  public function edit(News $news)
  {
    return view('news.edit', compact('news'));
  }

  /**
   * Update the specified news in storage.
   */
  public function update(Request $request, News $news)
  {
    $validator = Validator::make($request->all(), [
      'content' => 'required|string|min:10',
      'is_active' => 'boolean'
    ]);

    if ($validator->fails()) {
      return redirect()->back()
        ->withErrors($validator)
        ->withInput();
    }

    $news->update([
      'content' => $request->content,
      'is_active' => $request->has('is_active')
    ]);

    return redirect()->route('news.index')
      ->with('success', 'تم تحديث الخبر بنجاح');
  }

  /**
   * Remove the specified news from storage.
   */
  public function destroy(News $news)
  {
    $news->delete();
    return redirect()->route('news.index')
      ->with('success', 'تم حذف الخبر بنجاح');
  }

  /**
   * Toggle the active status of the news.
   */
  public function toggleStatus(News $news)
  {
    $news->update(['is_active' => !$news->is_active]);

    $status = $news->is_active ? 'مفعل' : 'معطل';
    return redirect()->route('news.index')
      ->with('success', "تم $status الخبر بنجاح");
  }
}
