<?php

namespace App\Http\Controllers;

use App\DataTables\Agents\TransactionsDataTable;
use App\DataTables\TransactionDataTable;
use App\Services\StatementPdfService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class StatementController extends Controller
{
  public function  index(TransactionDataTable $dataTable, Request $req)
  {
    $account_code = $req->account_code;
    return $dataTable->render('Statements.account_statements', compact('account_code'));
  }


  /**
   * Handle AJAX requests for TransactionDataTable
   */
  public function data(Request $request)
  {
    $dataTable = new TransactionDataTable();

    // Set filters from request
    $dataTable->filters = [
      'account' => $request->get('account'),
      'from_date' => $request->get('from_date'),
      'to_date' => $request->get('to_date'),
      'type' => $request->get('type'),
      'status' => $request->get('status'),
    ];

    return $dataTable->ajax();
  }

  ///////////////////////////////Agent/////////////////////////////////////

  public function  Agent_index(TransactionsDataTable $dataTable, Request $req)
  {
    $agent = auth()->guard('agent')->user()->agent->rootAgent();
    $accounts = "agent";
    $cashAccount = $agent->getAgentCashAccount();

    $account_code = $cashAccount;
    return $dataTable->render('AgentPanel.Statements.index', compact('account_code', 'accounts'));
  }
  public function  Cashier_index(TransactionDataTable $dataTable, Request $req)
  {
    $account_code = $req->account_code;
    $accounts = "cashier";
    return $dataTable->render('AgentPanel.Statements.index', compact('account_code', 'accounts'));
  }


  /**
   * Export statement to PDF
   */
  public function exportPdf(Request $request)
  {
    try {
      $filters = $request->all();
      Log::info('PDF Export Request - Filters:', $filters);

      if (empty($filters['account']) || empty($filters['currency'])) {
        Log::warning('PDF Export - Missing required filters');
        return response()->json(['error' => 'يرجى تحديد الحساب والعملة أولاً'], 400);
      }

      $statementDetails = $this->getStatementDetails($filters);
      Log::info('PDF Export - Statement Details:', $statementDetails);

      $pdfService = app(StatementPdfService::class);
      $mpdf = $pdfService->generateStatementPdf($filters);

      $filename = $this->generateSafeFilename($statementDetails);

      // Return the PDF as a download response
      return response($mpdf->Output($filename, 'S'), 200, [
        'Content-Type' => 'application/pdf',
        'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        'Content-Length' => strlen($mpdf->Output('', 'S')),
      ]);
    } catch (\Exception $e) {
      Log::error('Error exporting statement PDF: ' . $e->getMessage(), [
        'trace' => $e->getTraceAsString(),
        'filters' => $request->all()
      ]);
      return response()->json(['error' => 'حدث خطأ أثناء تصدير PDF: ' . $e->getMessage()], 500);
    }
  }

  /**
   * Get statement details for PDF export
   */
  protected function getStatementDetails($filters)
  {
    // This should integrate with your existing statement logic
    // For now, returning sample data structure
    return [
      'account_name' => $filters['account_name'] ?? 'حساب افتراضي',
      'opening_balance' => $filters['opening_balance'] ?? 0,
      'currency' => $filters['currency'] ?? 'TRY',
      'balance_type' => $filters['balance_type'] ?? 'مدين لنا',
      'current_balance' => $filters['current_balance'] ?? 0,
      'fromDate' => $filters['fromDate'] ?? Carbon::now()->subMonth()->format('Y-m-d'),
      'toDate' => $filters['toDate'] ?? Carbon::now()->format('Y-m-d'),
      'entry_count' => $filters['entry_count'] ?? 0,
      'account_code' => $filters['account_code'] ?? 'N/A',
      'outTransfers' => $filters['outTransfers'] ?? 0,
      'inTransfers' => $filters['inTransfers'] ?? 0,
      'total_debit' => $filters['total_debit'] ?? 0,
      'total_credit' => $filters['total_credit'] ?? 0,
      'currency_words' => $filters['currency_words'] ?? 'ليرة تركية',
      'account_type' => $filters['account_type'] ?? 'regular',
    ];
  }

  /**
   * Generates a safe filename by replacing invalid characters.
   */
  protected function generateSafeFilename($statementDetails)
  {
    // Format dates to avoid slashes
    $fromDate = isset($statementDetails['fromDate'])
      ? str_replace(['/', '\\'], '-', $statementDetails['fromDate'])
      : 'from';

    $toDate = isset($statementDetails['toDate'])
      ? str_replace(['/', '\\'], '-', $statementDetails['toDate'])
      : 'to';

    $filename = 'statement_' .
      ($statementDetails['account_code'] ?? 'account') . '_' .
      $fromDate . '_to_' . $toDate . '.pdf';

    // Replace any remaining invalid characters for filenames
    $filename = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $filename);

    return $filename;
  }
}
