<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Session;
use App\Helpers\RedisSessionHelper;

class CheckAgentStatus
{
  /**
   * Handle an incoming request.
   *
   * @param  \Illuminate\Http\Request  $request
   * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
   * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
   */
  public function handle(Request $request, Closure $next)
  {
    // Check both web and agent guards
    $user = null;
    $guard = null;

    if (Auth::guard('web')->check()) {
      $user = Auth::guard('web')->user();
      $guard = 'web';
    } elseif (Auth::guard('agent')->check()) {
      $user = Auth::guard('agent')->user();
      $guard = 'agent';
    }

    if ($user) {
      Log::info("CheckAgentStatus: User ID: {$user->id}, Guard: {$guard}, Agent ID: {$user->agent_id}");

      // Check if user has an agent_id
      if ($user->agent_id) {
        // Check if the agent is still active
        $agent = DB::table('agents')
          ->where('id', $user->agent_id)
          ->first();

        if (!$agent || !$agent->is_active) {
          // Agent is disabled or doesn't exist - logout the user
          Log::warning("CheckAgentStatus: User ID {$user->id} was logged out because their agent (ID: {$user->agent_id}) is disabled or doesn't exist");

          // Invalidate tokens
          DB::table('personal_access_tokens')
            ->where('tokenable_type', 'App\Models\User')
            ->where('tokenable_id', $user->id)
            ->delete();

          // Clear Redis sessions using the helper
          RedisSessionHelper::clearUserSessions($user->id);

          // Logout the user from the appropriate guard
          Auth::guard($guard)->logout();

          // Clear session
          $request->session()->invalidate();
          $request->session()->regenerateToken();

          // Redirect with error message
          if ($request->expectsJson()) {
            return response()->json([
              'message' => __('Your account has been disabled because your agent account is no longer active.'),
              'code' => 'AGENT_DISABLED'
            ], 403);
          }

          // Redirect based on guard
          if ($guard === 'agent') {
            return redirect("/login")->withErrors([
              'email' => __('Your account has been disabled because your agent account is no longer active.')
            ]);
          } else {
            return redirect()->route('login')->withErrors([
              'email' => __('Your account has been disabled because your agent account is no longer active.')
            ]);
          }
        }
      }
    }

    return $next($request);
  }
}
