<?php

namespace App\LedgerReports;

use Abivia\Ledger\Messages\Report;
use Abivia\Ledger\Models\ReportData;
use Abivia\Ledger\Reports\AbstractReport;

use Abivia\Ledger\Models\LedgerDomain;
use Abivia\Ledger\Models\LedgerCurrency;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\JournalDetail;
use App\Models\Currency;
use App\Models\CustomLedgerAccount;
use App\Models\JournalDetailOverwrite;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AccountStatementReport extends AbstractReport
{
  protected Report $reportMessage;

  public function collect(Report $report): ReportData
  {
    $report->validate();
    $this->reportMessage = $report;

    $ledgerDomain = LedgerDomain::findWith($report->domain)->firstOrFail();
    $currency = LedgerCurrency::findOrBreaker($report->currency);

    $reportData = new AccountStatementData();
    $reportData->request = $report;
    $reportData->decimals = $currency->decimals;

    $account_code = $report->options['account'] ?? null;
    if ($account_code) {
      $account = CustomLedgerAccount::where("code", $account_code)->first();
      $reportData->account = $account;

      $from = $report->fromDate ?? '1900-01-01';
      $to = $report->toDate ?? now()->toDateString();




      // Get transactions within the date range
      $details = JournalDetailOverwrite::with('entry', 'account', 'entry.transfer', 'entry.currencyrelation', 'entry.transfer.user')
        ->whereHas('account', function ($q) use ($account) {
          $q->where('ledgerUuid', $account->ledgerUuid);
        })
        ->whereHas('entry', function ($q) use ($from, $to) {
          $q->where('currency', $this->reportMessage->currency)
            ->whereBetween('transDate', [$from, $to])->orderBy('transDate');
        })

        ->get();

      $reportData->lines = $details->sortBy(function ($detail) {
        return $detail->entry->transDate;
      })->values();
      $reportData->openingBalance =  $account->getBalanceByDate(Carbon::parse($from)->toDateString(), $this->reportMessage->currency);
      Log::info('Opening balance: ' . $reportData->openingBalance);

      return $reportData;
    } else {
      Log::error('Account code not provided in report options');
      throw new \Exception('Account code is required for account statement report.');
    }
  }

  public function prepare(ReportData $reportData): Collection
  {
    /** @var AccountStatementData $data */
    $data = $reportData;

    $balance = $data->openingBalance * (-1);
    $decimals = $data->decimals;
    $totalDebit = '0';
    $totalCredit = '0';

    $rows = collect();

    // Sort lines by transDate
    $sortedLines = $data->lines->sortBy(function ($detail) {
      return $detail->entry->transDate;
    });

    $counter = 1;
    $in_transfers = 0;
    $out_transfers = 0;
    foreach ($sortedLines as $detail) {
      $entry = $detail->entry;
      $account = $detail->account;
      Log::info('Account Model Info', [
        'model_class' => get_class($account),
        'table_name' => $account->getTable(),
        'connection' => $account->getConnectionName(),
        'primary_key' => $account->getKeyName()
      ]);
      $transfer = $entry->transfer;


      $user = $entry->user->name ?? 'اليوزر';
      $type = $entry->reference_type ?? '';
      $documentNumber = $entry->journalReferenceUuid;
      $description = $entry->description ?? '';
      $review = $entry->reviewed ?? 0;
      $debit = $detail->amount < 0 ? bcmul($detail->amount, '-1', $decimals) : '0';
      $credit = $detail->amount >= 0 ? $detail->amount : '0';


      $totalDebit = bcadd($totalDebit, $debit, $decimals);
      $totalCredit = bcadd($totalCredit, $credit, $decimals);
      // Calculate balance
      $amountChange = bcsub($debit, $credit, $decimals);
      $balance = bcadd($balance, $amountChange, $decimals);
      $undelivered_transfers = $account->agent?->rootAgent()?->undeliveredTransfers()->get();
      Log::info($undelivered_transfers);
      $corss_account = $transfer->sender_id == $account->agent?->id ? $transfer->receiver?->name : $transfer->sender?->name;
      $direction = $transfer->sender_id == $account->agent?->id ? "out" : "in";
      $debit !== '0' ? $out_transfers++ : $in_transfers++;
      $rows->push([
        '#' => $counter++,
        'id' => $entry->journalEntryId,
        'date' => $entry->transDate instanceof \Carbon\Carbon
          ? $entry->transDate
          : \Carbon\Carbon::parse($entry->transDate),
        'transfer_id' => $transfer->id ?? "",
        'user' => $transfer->user->name ?? "",
        'transfer_side' => $transfer?->user?->agent ? __('agent') : __("administrator"),
        'transfer_side_color' => $transfer?->user?->agent ? 'bg-primary' : 'bg-secondary',
        'account' => $account,
        'transfer_number' => $transfer->reference ?? "",
        'sender' => $transfer->sender->name ?? "",
        'receiver' => $corss_account,
        'send_fee' => $transfer->send_fee ?? "",
        'delivery_fee' => $transfer->receiver_fee ?? "",
        'type' => $transfer->type ?? "",
        'document' => $documentNumber,
        'debit' => $debit !== '0' ? $debit : '',
        'credit' => $credit !== '0' ? $credit : '',
        'client' => $detail->extra ?? '-',
        'description' => $transfer?->getDescription($direction),
        'amount' => $debit !== '0' ? $debit : $credit,
        'balance' => $balance,
        'currency' => $this->reportMessage->currency,
        //  "currency_name" => $entry->currencyrelation->name ?? '',
        'review' => $review,
        'transfer_type' => $transfer?->getTransferType($debit),
        'direction' => $debit !== '0' ? __('out') : __('in')
      ]);
    }

    $totalsRow = [
      '#' => $counter++,
      'id' => 0,
      'date' => '',
      'user' => '',
      'transfer_side' => '',
      'account' => $account,
      'type' => '',
      'document' => '',
      'debit' => $totalDebit,
      'credit' => $totalCredit,
      'client' => '',
      'description' => __('إجمالي الحركة'),
      'amount' => '',
      'balance' => '',
      'currency' => $this->reportMessage->currency,
      //   "currency_name" => Currency::getCurrencyName($this->reportMessage->currency),
      'review' => $review
    ];

    $rows->push($totalsRow);
    $data->totalDebit = $totalDebit;
    $data->totalCredit = $totalCredit;
    $data->entryCount = $rows->count() - 1;
    $data->outTransfers = $out_transfers;
    $data->inTransfers = $in_transfers;
    $data->currentBalance = $balance;
    $data->last_id = $counter++;

    Log::info($data->totalDebit . ' ' . $data->totalCredit . ' ' . $data->entryCount . ' ' . $data->inTransfers . ' ' . $data->outTransfers);

    return $rows;
  }
}
