<?php

namespace App\Livewire\AgentPanel\Transfer;


use App\Models\Agent;
use App\Models\Client;
use App\Models\Currency;
use App\Models\IdentityType;
use App\Models\LedgerApproval;
use App\Models\LedgerTransfer;
use App\Services\Transfer\TransferService;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Livewire\Component;
use OwenIt\Auditing\Events\AuditCustom;

class DeliverTransfer extends Component
{
  public ?int $transfer_id = null;
  public array $transfer = [];
  public $selected_transfer;
  public Collection $identity_types;
  public array $accounts = [];
  public array $recipient = ['identity_number' => null];
  public ?string $secret = null;
  public bool $enable_button = true;
  public array $attachments = [];
  public $add_client = false;
  public string $clientsSearchUrl = '/Client/Search';
  public string $searchUrl = '/Client/Search';
  public string $transferSearchURL = '/Transfers/Search';
  public $client;
  protected function getListeners(): array
  {
    return [
      'deliverTransfer' => 'deliverTransfer',
      'AddNewClientEvent' => 'AddNewClientEvent',
      "CancelAddClientEvent" => "CancelAddClientEvent",
      'getClientData' => 'getClientData',
    ];
  }
  public  function  getClientData($id)
  {
    $id = $id['id'];
    $client = Client::find($id);
    $this->client = $client;
    $this->recipient['recipient_name'] = $client->name;
    $this->recipient['recipient_phone'] = $client->phone;
    $this->recipient['sender_client_identity_number'] = $client->identity_number;
    $this->recipient['sender_client_id'] = $client->id ?? null;
    $this->recipient['sender_identity_number'] = $client->id ?? null;

    $this->transfer['recipient_client_id'] = $client->id;
    $this->transfer['recipient_id_number'] = $client->identity_number;
    $this->recipient['identity_number'] = $client->id;
    $this->add_client = false;
  }
  public function  CancelAddClientEvent()
  {
    $this->add_client = false;
  }
  public  function  AddNewClientEvent()
  {
    $this->add_client = true;
  }
  public function mount(?LedgerTransfer $transfer = null): void
  {
    $this->identity_types = IdentityType::all();
  }

  public function searchTransfer(): void
  {
    $reference = $this->transfer['reference'] ?? null;
    $secret = $this->transfer['secret'] ?? null;
    try {
      $rules = [
        'transfer.reference' => 'required|string',
        'transfer.secret' => 'required|string',
      ];
      $this->validate($rules);
    } catch (ValidationException $e) {
      $firstField = array_key_first($e->validator->errors()->messages());
      $firstErrorMessage = $e->validator->errors()->first($firstField);
      $this->dispatch('focus-error', ['field' => $firstField]);
      $this->dispatch("sweetalert:error", [
        'msg' => $firstErrorMessage,
        'title' => __("alerts.error"),
        'type' => "error"
      ]);

      throw $e; // rethrow so Livewire still shows errors
    } catch (\Exception $e) {
      $this->error(__('transfers.validation_failed'));
      return;
    }
    if (!$reference) {
      $this->error(__('transfers.transfer_not_found'));
      return;
    }

    $reference = trim($reference);
    $transfer = LedgerTransfer::withoutGlobalScopes()->with(['client', 'sender', 'receiver'])
      ->where(function ($query) use ($reference) {
        $query->where('reference', $reference)
          ->orWhere('id', $reference);
      })
      ->where('secret', $secret) // add your secret condition
      ->first();


    if (!$transfer) {
      $this->selected_transfer = null;
      $this->error(__('transfers.transfer_not_found'));
      return;
    }
    switch ($transfer->status) {
      case 'pending':
        // Handle pending case
        break;
      case 'completed':
        $this->error(__('transfers.transfer_already_delivered'));
        $this->enable_button = false;
        return;
        break;
      case 'cancelled':
        $this->error(__('transfers.transfer_already_cancelled'));
        return;
      case 'reverse':
        $this->error(__('transfers.transfer_already_reversed'));
        return;
      case 'deleted':
        $this->error(__('transfers.transfer_already_deleted'));
        return;
      default:
        $this->error(__('transfers.unknown_transfer_status'));
        return;
    }
    if ($transfer) {

      if ($transfer->sender_id == auth()->guard('agent')->user()->agent->rootAgent()->id) {
        $this->dispatch("sweetalert:error", [
          'msg' => __('transfers.can_not_receive_transfer_you_are_sender'),
          'title' => __("Success"),
          'type' => "success"
        ]);
        return;
      }
      $this->selected_transfer = $transfer;
    }
    $this->transfer = $transfer->toArray();
    $this->transfer['amount_number'] = NoStylePriceFormat($this->transfer['delivery_amount'], $this->transfer['delivery_currency']);
    $this->transfer['written_amount'] = toCurrencyWords($this->transfer['delivery_amount'], $this->transfer['delivery_currency']);
    $this->transfer['reason'] = $this->selected_transfer->reason->name ?? " ";
    if ($transfer->status === 'Completed') {
      $this->error(__('transfers.transfer_already_delivered'));
      $this->enable_button = false;
    }

    $this->accounts = $transfer->type !== 'transfer'
      ? LedgerApproval::GetDestinationAgents()->toArray()
      : LedgerTransfer::GetDestinationAgents('delivering')->toArray();


    $this->dispatch('refresh-select2', [
      'target' => 'transfer.receiver_id',
      'options' => $this->formatAccountOptions($this->accounts),
    ]);
  }

  public function searchReceiver(): void
  {
    $this->validate([
      'recipient.identity_number' => 'required|string',
    ]);

    $identityNumber = $this->recipient['identity_number'] ?? null;

    $client = Client::query()
      ->where('phone', $identityNumber)
      ->orWhere('identity_number', $identityNumber)
      ->orWhere('name', $identityNumber)
      ->orWhere('id', $identityNumber)
      ->first();

    if (!$client) {
      $this->error(__('transfers.client_not_found'));
      return;
    }

    $this->transfer['recipient_client_id'] = $client->id;
    $this->transfer['recipient_id_number'] = $client->identity_number;

    $this->dispatch('LoadClientData', ['id' => $client->id]);
  }

  public function deliverTransfer()
  {
    $transfer = $this->selected_transfer;

    if (!$transfer || empty($this->transfer['id'])) {
      $this->error(__('transfers.transfer_not_found'));
      return;
    }

    if (in_array($transfer->status, ['reverse', 'cancelled'], true)) {
      $this->error(__('transfers.transfer_already_cancelled'));
      return;
    }

    if ($transfer->type === 'approval') {
      $rules = LedgerApproval::getDeliveryRules();
    } else {
      if ($transfer->status === 'completed') {
        $this->error(__('transfers.transfer_already_delivered'));
        return;
      }
      $rules = LedgerTransfer::getDeliveryRules();

      if (empty($this->transfer['recipient_client_id'])) {
        $this->error(__('transfers.recipient_not_found'));
        return;
      }
    }



    if ($this->transfer['secret'] !== $transfer->secret) {
      $this->error(__('transfers.invalid_secret'));
      return;
    }

    try {
      $this->validate($rules);

      $this->processDelivery($transfer);
      $this->success(__('transfers.transfer_delivered_successfully'));
      $this->dispatch('RefreshDashboardStatistics');
      $this->dispatch("ShowDeliveryReceipt", ['transfer_id' => $transfer->id]);


      $this->ResetForm();
    } catch (\Throwable $e) {
      Log::error($e);
      $this->error($e->getMessage());
    }
  }
  public  function  ResetForm()
  {
    $this->transfer = [];
    $this->selected_transfer = null;
    $this->recipient = [];
    $this->client = null;
  }

  private function processDelivery(LedgerTransfer $transfer): void
  {

    $transfer->receiver_id = auth()->guard('agent')->user()->agent->rootAgent()->id;
    $transfer->recipient_client_id = $this->recipient['sender_client_id'] ?? null;
    $transfer->recipient_id_number = $this->recipient['sender_identity_number'] ?? null;
    $transfer->disableAuditing();

    (new TransferService())->DeliverTransfer($transfer);

    $transfer->status = 'Completed';

    $transfer->delivered_at = now();
    $transfer->save();

    $this->logAudit($transfer);

    $this->dispatch('saveMedia', LedgerTransfer::class, $transfer->id);
    $this->dispatch('refreshYajraDatatable', ['table' => 'transfer-table']);
  }

  private function logAudit(LedgerTransfer $transfer): void
  {
    $recipient = Client::find($transfer->recipient_client_id);

    $transfer->enableAuditing();
    $transfer->auditEvent = 'Delivered';
    $transfer->isCustomEvent = true;
    $transfer->auditCustomOld = [];
    $transfer->auditCustomNew = [
      'recipient_client_id' => $transfer->recipient_client_id,
      'recipient_id_number' => $transfer->recipient_id_number,
      'recipient_name' => $recipient->name ?? '',
      'recipient_phone' => $recipient->phone ?? '',
      'status' => 'Completed',
    ];

    Event::dispatch(new AuditCustom($transfer));
    $transfer->disableAuditing();
  }

  private function formatAccountOptions(array $accounts): array
  {
    return array_map(function ($account) {
      $name = $account['name'] ?? '';
      return [
        'id' => $account['id'],
        'text' => $account['text'] ?? $name,
      ];
    }, $accounts);
  }

  private function error(string $message): void
  {
    $this->dispatch('sweetalert:error', [
      'msg' => $message,
      'title' => 'Error',
      'type' => 'error',
    ]);
  }

  private function success(string $message): void
  {
    $this->dispatch('sweetalert:success', [
      'msg' => $message,
      'title' => 'Success',
      'type' => 'success',
    ]);
  }

  public function updatedRecipientIdentityNumber(): void
  {
    $this->searchReceiver();
  }

  public function updatedTransferReference(): void
  {
    $this->searchTransfer();
  }
  public function render()
  {
    return view('livewire.agent-panel.transfer.deliver-transfer');
  }
}
