<?php

namespace App\Livewire\Dashboard;

use App\Models\LedgerTransfer;
use App\Models\LedgerApproval;
use App\Models\LedgerBooking;
use App\Models\LedgerExchange;
use Illuminate\Support\Facades\Log;
use Livewire\Component;

class TransferChart extends Component
{
  public $selectedDateRange = '10'; // Default to last 10 days
  public $selectedTransferType = 'transfers'; // Default to all types
  public $dateFrom;
  public $dateTo;

  public $categories = [];
  public $series = [];

  protected $listeners = ['refreshChart' => 'getTransferChartData'];

  public function mount()
  {
    $this->getTransferChartData();
  }

  public function getTransferChartData()
  {
    $query = null;
    Log::info($this->selectedTransferType);
    // Base query based on selected type
    switch ($this->selectedTransferType) {
      case 'transfers':
        $query = LedgerTransfer::transfer();
        break;
      case 'approvals':
        $query = LedgerApproval::query();
        break;
      case 'bookings':
        $query = LedgerBooking::query()->where("type", "booking")->whereNotIn("status", ["cancelled", "deleted"]);
        break;
      case 'exchanges':
        $query = LedgerExchange::query();
        break;
      case 'all':
      default:
        // For union approach, we need to handle this differently
        // Let's use a simpler approach by querying each type separately and merging
        $query = LedgerTransfer::transfer();
        return;
    }

    // Apply date filtering
    $this->applyDateFilter($query);

    // Get daily data
    $monthly = $query->selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as day, type, SUM(amount) as total, COUNT(id) as count")
      ->groupBy('day', 'type')
      ->orderBy('day')
      ->get()
      ->groupBy('type');

    $this->categories = $monthly->first()?->pluck('day')->unique()->values()->toArray() ?? [];

    $this->series = $monthly->map(function ($items, $type) {
      return [
        'name' => __($type),
        'type' => 'column',
        'data' => collect($items)->pluck('count')->map(fn($v) => $v)->toArray()
      ];
    })->values()->toArray();

    $this->dispatch('TransferChartDataUpdated', [
      'categories' => $this->categories,
      'series' => $this->series,
    ]);
    Log::info($this->categories);
    Log::info($this->series);
  }

  // Add this new method to handle date filtering
  private function applyDateFilter($query)
  {
    switch ($this->selectedDateRange) {
      case '7':
        $query->where('created_at', '>=', now()->subDays(7));
        break;
      case '10':
        $query->where('created_at', '>=', now()->subDays(10));
        break;
      case '30':
        $query->where('created_at', '>=', now()->subDays(30));
        break;
      case '90':
        $query->where('created_at', '>=', now()->subDays(90));
        break;
      case 'year':
        $query->where('created_at', '>=', now()->startOfYear());
        break;
    }
  }

  // Update the getCombinedData method
  private function getCombinedData()
  {

    // Query each type separately
    $transfers = LedgerTransfer::transfer();
    $this->applyDateFilter($transfers);
    $transfers = $transfers->selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as day, 'transfer' as type, SUM(amount) as total, COUNT(id) as count")
      ->groupBy('day')
      ->get();

    $approvals = LedgerApproval::query();
    $this->applyDateFilter($approvals);
    $approvals = $approvals->selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as day, 'approval' as type, SUM(amount) as total, COUNT(id) as count")
      ->groupBy('day')
      ->get();

    $bookings = LedgerBooking::query()->where("type", "booking");
    $this->applyDateFilter($bookings);
    $bookings = $bookings->selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as day, 'booking' as type, SUM(amount) as total, COUNT(id) as count")
      ->groupBy('day')
      ->get();

    $exchanges = LedgerExchange::query();
    $this->applyDateFilter($exchanges);
    $exchanges = $exchanges->selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as day, 'exchange' as type, SUM(amount) as total, COUNT(id) as count")
      ->groupBy('day')
      ->get();

    // Combine all data
    $allData = $transfers->concat($approvals)->concat($bookings)->concat($exchanges);

    // Group by type and prepare series
    $groupedData = $allData->groupBy('type');

    // Get unique days for categories
    $this->categories = $allData->pluck('day')->unique()->sort()->values()->toArray();

    // Prepare series data
    $this->series = $groupedData->map(function ($items, $type) {
      // Create a map of day -> count for this type
      $dayCountMap = $items->pluck('count', 'day')->toArray();

      // Fill in missing days with 0
      $data = collect($this->categories)->map(function ($day) use ($dayCountMap) {
        return $dayCountMap[$day] ?? 0;
      })->toArray();

      return [
        'name' => __($type),
        'type' => 'column',
        'data' => $data
      ];
    })->values()->toArray();

    $this->dispatch('TransferChartDataUpdated', [
      'categories' => $this->categories,
      'series' => $this->series,
    ]);
  }

  // Remove these methods since we no longer need custom date handling
  // public function updatedDateFrom()
  // public function updatedDateTo()
  public function updatedSelectedDateRange()
  {

    $this->getTransferChartData();
  }

  public function updatedSelectedTransferType()
  {
    $this->getTransferChartData();
  }

  public function updatedDateFrom()
  {
    if ($this->dateFrom && $this->dateTo) {
      $this->getTransferChartData();
    }
  }

  public function updatedDateTo()
  {
    if ($this->dateFrom && $this->dateTo) {
      $this->getTransferChartData();
    }
  }

  public function render()
  {
    return view('livewire.dashboard.transfer-chart');
  }
}
