<?php

namespace App\Livewire\Exchange;

use App\Models\Currency;
use App\Models\ExchangePrice;
use Illuminate\Support\Facades\Log;
use Livewire\Component;

class ExchangePrices extends Component
{
    public array $rates = [];
    public array $groupedRates = [];
    public ?string $selectedFrom = null;
    public ?string $selectedTo = null;
    public ?string $selectedType = null;
    public ?float $selectedRate = null;

    public function mount()
    {
        $this->refreshRates();
    }

    public function refreshRates()
    {
        try {
            // 1. Get base currency safely
            $baseCurrency = Currency::where('is_default', 1)->first();

            if (!$baseCurrency) {
                $this->groupedRates = [];
                Log::warning('No default currency set');
                return;
            }

            $baseCurrencyCode = $baseCurrency->code;

            // 2. Get prices with eager loading
            $prices = ExchangePrice::where('type', 'transfer')
                ->with(['fromCurrency', 'toCurrency'])
                ->get();

            if ($prices->isEmpty()) {
                $this->groupedRates = [];
                return;
            }

            $grouped = [];

            foreach ($prices as $price) {
                // 3. Skip if currencies are not loaded
                if (!$price->fromCurrency || !$price->toCurrency) {
                    continue;
                }

                $from = $price->from_currency;
                $to = $price->to_currency;

                // 4. Create sorted pair key
                $pair = [$from, $to];
                sort($pair);
                $pairKey = implode('/', $pair);

                // Initialize pair if not exists
                if (!isset($grouped[$pairKey])) {
                    $grouped[$pairKey] = [
                        'from' => $pair[0],
                        'to' => $pair[1],
                        'from_avatar' => $price->fromCurrency->getAvatar(),
                        'to_avatar' => $price->toCurrency->getAvatar(),
                        'buy' => null,
                        'sell' => null,
                        'buy_factor' => null,
                        'sell_factor' => null
                    ];
                }

                // 5. Set buy/sell rates
                if ($price->to_currency === $baseCurrencyCode) {
                    $grouped[$pairKey]['buy'] = $price->rate;
                    $grouped[$pairKey]['buy_factor'] = $price->operation;
                }

                if ($price->from_currency === $baseCurrencyCode) {
                    $grouped[$pairKey]['sell'] = $price->rate;
                    $grouped[$pairKey]['sell_factor'] = $price->operation;
                }
            }

            $this->groupedRates = $grouped;
        } catch (\Exception $e) {
            Log::error('Error refreshing rates: ' . $e->getMessage());
            $this->groupedRates = [];
        }
    }

    public function setExchange($from, $to, $rate, $type, $factor)
    {
        $this->selectedFrom = $from;
        $this->selectedTo = $to;
        $this->selectedRate = $rate;
        $this->selectedType = $type;

        $this->dispatch('SetExchange', [
            'from' => $from,
            'to' => $to,
            'rate' => $rate,
            'type' => $type,
            'factor' => $factor
        ]);
    }

    public function render()
    {
        return view('livewire.exchange.exchange-prices', [
            'groupedRates' => $this->groupedRates ?: []
        ]);
    }
}
