<?php

namespace App\Livewire\Exchange;

use App\Exceptions\InvalidExchangeRateException;
use App\Models\Client;
use App\Models\Currency;
use App\Models\LedgerTransfer;
use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\Transfer\ExchangeService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;

class MultiCurrency extends Component
{
    public array $items = [];
    public  $recipient = [];
    public $exchange = [];
    public  $type = "buy";
    public $currency;
    public  $currencies = [];
    public string $createExchangeMessage = '';
    public bool $canCreateExchange = false;
    protected  function  getListeners()
    {
        return [
            'SetExchangeAgent' => 'SetExchangeAgent',
            'SetExchange' => 'setExchange'
        ];
    }
    public function  SetExchangeAgent($data)
    {


        $this->exchange['sender_id'] = (int) $data['agent_id'];
        $this->exchange['receiver_id'] = (int) $data['agent_id'];
    }
    public function setExchange($data)
    {
        Log::info("Setting exchange with data: ", $data);

        $this->exchange['currency'] = $data['from'];
        $this->exchange['delivery_currency'] = $data['to'];
        $this->exchange['rate'] = $data['rate'];
        $this->exchange['rate_factor'] = $data['factor'] == "/" ? "divide" : "multiply";

        //    $this->Calculate();
    }

    public function updated($key, $value)
    {
        Log::info("Exchange updated: ", $this->exchange);
        //  $this->exchange['rate_factor'] = getCurrencyFactor($this->currency, $this->exchange['delivery_currency'], 'balance');
        $this->calculateAll();
    }
    public function updatedItems($value, $key)
    {

        $rules = [
            'currency' => 'required',
            'type' => 'required'
        ];
        $this->validate($rules);
        [$index, $field] = explode('.', $key);
        $item = $this->items[$index];
        $factor = $item['factor'] ?? "multiply";
        $converter = new CurrencyConverter();

        $from_amount = $item['amount'];
        $rate = $item['rate'];
        $item_currency = $item['currency'];
        Log::info($factor);

        // Determine direction of conversion based on type (buy/sell)
        if ($this->type === 'buy') {
            $from_currency = $this->currency;          // base/system currency
            $to_currency = $item_currency;             // selected currency in item
            $factor === 'multiply' ? '*' : '/';
        } else { // sell
            $from_currency = $item_currency;
            $to_currency = $this->currency;
            $factor === 'multiply' ? '/' : '*';
        }
        if ($field === 'currency') {
            $factor = getCurrencyFactor($from_currency, $to_currency, 'balance');
            $rate = exchange_rate($from_currency, $to_currency,   'balance');
            $factor = $factor == "multiply" ? "divide" : "multiply";
            $this->items[$index]['factor'] = $factor;
            $this->items[$index]['rate'] = $rate;
            $item['factor'] = $factor;
            $item['rate'] = $rate;
        }


        $factor === 'multiply' ? '/' : '*';
        // Ensure required fields exist and are numeric
        if (
            isset($item['amount'], $item['rate'], $item['currency']) &&
            is_numeric($item['amount']) &&
            is_numeric($item['rate'] && isset($item["factor"]))
        ) {



            try {
                $converted = $converter->convert(
                    $from_amount,
                    $from_currency,
                    $to_currency,
                    $this->currencies,
                    'balance',
                    $rate,
                    $factor
                );

                $this->items[$index]['delivery_amount'] = round($converted, 2);
                Log::info("Conversion successful for item $index: " . $converted);
            } catch (\Throwable $e) {
                $this->items[$index]['delivery_amount'] = 0;
                Log::error("Conversion failed for item $index: " . $e);
            }
        }
    }

    public function calculateAll()
    {
        $converter = new CurrencyConverter();

        foreach ($this->items as $index => $item) {
            $amount = $item['amount'] ?? 0;
            $rate = $item['rate'] ?? null;
            $currency = $item['currency'] ?? null;
            $factor = $item['factor'] ?? "multiply";
            if (!$amount || !$rate || !$currency) {
                continue;
            }

            if ($this->type === 'buy') {
                $from_amount = $amount;
                $from_currency = $currency; // system currency
                $to_currency = $this->currency;
                $factor =  $factor === 'multiply' ? '*' : '/';
            } else { // sell
                $from_amount = $amount;
                $from_currency = $this->currency;
                $to_currency = $currency; // system currency
                $factor =  $factor === 'multiply' ? '*' : '/';
            }

            try {
                $converted = $converter->convert(
                    $from_amount,
                    $from_currency,
                    $to_currency,
                    $this->currencies,
                    'balance',
                    $rate,
                    $factor
                );

                // Save result back to items
                $this->items[$index]['delivery_amount'] = round($converted, 2);
                Log::info("Conversion successful for item $index: " . $converted);
            } catch (\Throwable $e) {
                Log::error("Currency conversion error: " . $e);
                $this->items[$index]['delivery_amount'] = 0;
            }
        }
    }

    public function Calculate($from_amount = null, $from_currency = null, $to_currency = null, $rate, $factor)
    {


        $converter = new CurrencyConverter();
        $to_amount = $converter->convert($from_amount, $from_currency,  $this->currency,  $this->currencies, 'balance', $rate, $factor == 'multiply' ? '*' : '/');


        $this->exchange['delivery_amount'] = $to_amount;
    }
    public function mount()
    {
        $this->currencies = Currency::get();
        $this->items = [
            ['currency' => '', 'amount' => '', 'rate' => '', 'factor' => 'multiply'],
        ];
        $response = Gate::inspect('create', \App\Models\LedgerExchange::class);

        $this->canCreateExchange = $response->allowed();
        $this->createExchangeMessage = $response->message() ?? '';
    }

    protected function buildExchangeFromItem(array $item, int $index, $ref): array
    {

        $exchange = [];
        $exchange['reference'] = $ref . '-' . ($index + 1);
        $exchange['sender_id'] = $this->exchange['sender_id'];
        $exchange['receiver_id'] = $this->exchange['receiver_id'];
        $exchange['client_id'] = $this->exchange['client_id'] ?? null;
        $exchange['client_sender_id_number'] = $this->exchange['client_sender_id_number'] ?? null;
        $exchange['type'] = 'exchange';

        if ($this->type == "buy") {
            $exchange['currency'] = $this->currency;
            $exchange['delivery_currency'] = $item['currency'];
            $exchange['amount'] = $item['delivery_amount'];
            $exchange['delivery_amount'] =  $item['amount'];
            $exchange['rate'] = $item['rate'];
            $exchange['rate_factor'] = "multiply";
        } else {
            $exchange['currency'] = $item['currency'];
            $exchange['delivery_currency'] = $this->currency;
            $exchange['amount'] = $item['amount'];
            $exchange['delivery_amount'] = $item['delivery_amount'];
            $exchange['rate'] = $item['rate'];
            $exchange['rate_factor'] = "divide";
        }

        return $exchange;
    }

    public function addItem()
    {
        $this->items[] = ['currency' => '', 'amount' => '', 'rate' => '', 'factor' => 'multiply'];
    }

    public function removeItem($index)
    {
        unset($this->items[$index]);
        $this->items = array_values($this->items); // reindex
    }
    public function getCurrencyWordProperty()
    {
        $currency_word = $this->getAmountInWords($this->getTotalAmountProperty(), $this->currency);
        $text = "علينا ";
        $html = "<span class='badge bg-success'>";
        if ($this->type == "buy") {
            $text = "لنا ";
            $html .= $text;
        } else {
            $html = "<span class='badge bg-danger'>";
            $text = "علينا ";
            $html .= $text;
        }
        Log::info($currency_word);
        $html .= PriceFormat($this->getTotalAmountProperty(), $this->currency);
        return $html . "$currency_word</span>";;
    }
    public function getAmountInWords($amount, $currency)
    {
        if ($amount && $currency) {
            return toCurrencyWords(
                floatval($amount),
                $currency,
                app()->getLocale()
            );
        }
        return '';
    }
    public function getTotalAmountProperty()
    {

        return collect($this->items)
            ->filter(fn($item) => isset($item['amount'], $item['rate']) && is_numeric($item['amount']) && is_numeric($item['rate']))
            ->sum(fn($item) => $item['delivery_amount']);
    }
    public function  searchClient()
    {
        $recipent = $this->recipient;
        $this->validate([
            'recipient.sender_identity_number' => 'required',
        ]);

        $client = Client::where('phone', '=', $recipent['sender_identity_number'])->orWhere('identity_number', '=', $recipent['sender_identity_number'])->orWhere('name', '=', $recipent['sender_identity_number'])->first();
        if ($client) {
            $this->recipient = $client->toArray();
            $this->recipient['sender_client_id'] = $client->id;
            $this->recipient['sender_identity_number'] = $client->identity_number;
            $this->exchange['client_id'] = $client->id;
            $this->exchange['client_sender_id_number'] =  $this->recipient['sender_identity_number'];
            $this->dispatch("LoadClientData", ['id' => $client->id]);
        } else {
            $this->dispatch("sweetalert:error", ['msg' => "No Client Found", "title" => "Error", "type" => "error"]);
        }
    }

    public function save()
    {

        if (!isset($this->exchange['sender_id'])) {
            $this->dispatch("sweetalert:error", ['msg' => __('exchange.select_sender'), 'title' => 'Error']);
            return;
        }
        if (!isset($this->exchange['client_id'])) {
            $this->dispatch("sweetalert:error", ['msg' => __('exchange.select_client'), 'title' => 'Error']);
            return;
        }
        $rules = [
            'type' => 'required',
            'currency' => 'required',
            'exchange.sender_id' => 'required',
            'recipient.sender_identity_number' => 'required',
            'exchange.receiver_id' => 'nullable',
            'exchange.client_id' => 'required',
            'items' => 'required|array|min:1',
            'items.*.currency' => 'required|string|in:USD,TRY,SYP', // Add allowed currencies
            'items.*.amount' => 'required|numeric|min:0.01',
            'items.*.rate' => 'required|numeric|min:0.0001',

        ];
        $this->validate($rules);
        //   $data['reference'] = $this->generateRandomTransferNumber();
        $ref =  generateReferenceNumber("EXG");
        foreach ($this->items as $index => $item) {
            $exchangeData = $this->buildExchangeFromItem($item, $index, $ref);
            $exchangeData['client_id']  = $this->exchange['client_id'];
            $exchangeData['client_sender_id_number'] =  $this->recipient['sender_identity_number'];
            DB::beginTransaction();
            try {
                $ex = LedgerTransfer::create($exchangeData);
                $service = new ExchangeService();
                $service->InsertExchangeEntries($ex);
                DB::commit();
                $this->dispatch("sweetalert:success", [
                    'msg' => 'Exchange saved successfully',
                    'title' => 'Success'
                ]);
            } catch (\Throwable $e) {
                DB::rollBack();
                Log::error('Exchange Save Failed: ' . $e->getMessage());
                $this->dispatch("sweetalert:error", [
                    'msg' => 'Exchange failed: ' . $e->getMessage(),
                    'title' => 'Error'
                ]);
            }
        }
    }
    public function render()
    {
        return view('livewire.exchange.multi-currency');
    }
}
