<?php

namespace App\Livewire;

use Livewire\Component;


use Illuminate\Support\Facades\Bus;
use Illuminate\Bus\Batch;
use Throwable;
use App\Exports\OutgoingTransfersExport;
use App\Exports\IncomingTransfersExport;
use App\Exports\LedgerTransferExport;
use App\Jobs\ExportToExcelJob;
use App\Models\ReportExportLog;
use Illuminate\Support\Facades\Storage;

class ReportExport extends Component
{
    public $reportType = 'outgoing';
    public $fromDate;
    public $toDate;
    public $batchId;
    public $progress = 0;
    public $filePath;

    public $logId;

    public function export()
    {
        $columnDefinitions = match ($this->reportType) {
            'outgoing' => [
                'reference' => 'Transfer Ref',
                'sender.name' => 'Sender',
                'amount' => 'Amount',
                'currency' => 'Currency',
                'sender_client.name' => 'Client Name',
                'sender_client.address' => 'Sender Address',
                'sender_client.address.country' => 'Country',
                'sender_client.IdentityType.name' => 'Identity Type',
                'sender_client.identity_number' => 'Identity Number',
                'recipient_client.name' => 'Recipient Name',
                'recipient_client.address' => 'Recipient Address',
                'receiver.name' => 'Receiver',
                'reason' => 'Reason',

                'created_at' => 'Date',
            ],
            'incoming' => [
                'reference' => 'Transfer Ref',
                'receiver.name' => 'Receiver',
                'recipient_client.name' => 'Recipient Name',
                'delivery_amount' => 'Received Amount',
                'delivery_currency' => 'Currency',
                'created_at' => 'Date',
            ],
            default => []
        };

        // Define relations and mapping function
        [$columns, $relations] = match ($this->reportType) {

            'outgoing' => [
                [
                    'Transfer Ref',
                    'Sender',
                    'Amount',
                    'Currency',
                    'Client Name',
                    'Sender Address',
                    'Country',

                    'Identity Type',
                    'Identity Number',

                    'Recipient Name',
                    'Recipient Address',
                    'Receiver',
                    'Reason',
                    'Date',
                ],
                ['sender', 'sender_client', 'recipient_client', 'reason']
            ],
            'incoming' => [
                [
                    'Transfer Ref',
                    'Receiver',
                    'Recipient Name',
                    'Received Amount',
                    'Currency',
                    'Date',
                ],
                ['receiver', 'sender_client', 'recipient_client']
            ],
            default => [[], []]
        };

        $export = new LedgerTransferExport(
            $this->fromDate,
            $this->toDate,
            $this->reportType,
            $columns,
            $relations
        );

        $export = new LedgerTransferExport(
            $this->fromDate,
            $this->toDate,
            $this->reportType,
            array_values($columnDefinitions), // only the headings

            $relations
        );

        $fileName = $this->reportType . '_report_' . now()->format('Ymd_His') . '.xlsx';
        $this->filePath = 'exports/' . $fileName;

        $log = ReportExportLog::create([
            'type' => $this->reportType,
            'from_date' => $this->fromDate,
            'to_date' => $this->toDate,
            'file_path' => $this->filePath,
            'status' => 'pending',
        ]);

        $this->logId = $log->id;

        $batch = Bus::batch([
            new ExportToExcelJob($export, $this->filePath)
        ])->then(function (Batch $batch) use ($log) {
            $log->update(['status' => 'completed']);
        })->catch(function (Batch $batch, Throwable $e) use ($log) {
            $log->update(['status' => 'failed']);
        })->dispatch();

        $this->batchId = $batch->id;
    }


    public function checkBatch()
    {
        if (!$this->batchId) return;

        $batch = Bus::findBatch($this->batchId);

        if ($batch) {
            $this->progress = $batch->progress();
        }
    }
    public function render()
    {
        return view('livewire.report-export');
    }
}
