<?php

namespace App\Livewire\Statements;

use Carbon\Carbon;
use Livewire\Component;
use Abivia\Ledger\Messages\Report;
use Abivia\Ledger\Http\Controllers\ReportController;
use App\LedgerReports\AccountStatementReport;
use App\Models\Currency;

class Details extends Component
{
  public $account_code;
  public array $totals = [
    'opening_balance' => 0,
    'total_debit' => 0,
    'total_credit' => 0,
    'entry_count' => 0,
    'outTransfers' => 0,
    'inTransfers' => 0,
    'fromDate' => '',
    'toDate' => '',
    'currency' => '',
    'net_balance' => 0,
    'balance_type' => '',
    'currency_words' => '',
    'account_name' => '',
    'account_code' => '',
    'current_balance' => 0
  ];

  public array $filters = [];

  protected $listeners = ['filtersApplied' => 'updateFilters'];

  public function mount($totals = [])
  {
    if (!empty($totals)) {
      $this->totals = $totals;
    }
  }

  public function updateFilters($filters)
  {

    $this->filters = $filters['filters'];
    $this->refreshTotals();
  }

  public function refreshTotals()
  {

    try {


      $report = Report::fromArray([
        'name' => 'accountStatement',
        'currency' => $this->filters['currency'] ?? 'TRY',
        'fromDate' => $this->filters['fromDate'] ?? Carbon::now()->subMonth()->toDateString(),
        'toDate' => $this->filters['toDate'] ?? Carbon::now()->toDateString(),
        'options' => [
          'account' => $this->filters['account'] ?? null,
        ],
      ]);

      $reportClass = new AccountStatementReport();

      // Step 1: collect report data
      $reportData = $reportClass->collect($report);

      // Step 2: prepare (populate rows + totals)
      $rows = $reportClass->prepare($reportData);
      $difference = $reportData->totalDebit - $reportData->totalCredit;

      if ($difference > 0) {
        $balanceType = 'لنا';
        $balanceAmount = $difference;
      } elseif ($difference < 0) {
        $balanceType = 'علينا';
        $balanceAmount = abs($difference); // Absolute value (removes negative sign)
      } else {
        $balanceType = 'صفر';
        $balanceAmount = 0;
      }
      $currencyWords = toCurrencyWords(
        floatval(abs($reportData->currentBalance ?? 0)),
        $this->filters['currency'],
        app()->getLocale()
      );
      // Step 3: now you have both rows and totals
      $this->totals = [
        'opening_balance' => $reportData->openingBalance,
        'total_debit' => $reportData->totalDebit,
        'total_credit' => $reportData->totalCredit,
        'entry_count' => $reportData->entryCount,
        'outTransfers' => $reportData->outTransfers,
        'inTransfers' => $reportData->inTransfers,
        'net_balance' => $balanceAmount,
        'current_balance' => $reportData->currentBalance,
        'balance_type' => $balanceType,
        'fromDate' => $this->filters['fromDate'] ?? Carbon::now()->subMonth()->toDateString(),
        'toDate' => $this->filters['toDate'] ?? Carbon::now()->toDateString(),
        'currency' => Currency::getCurrencyName($this->filters['currency'] ?? 'TRY'),
        'currency_words' => $currencyWords,
        'account_name' => $reportData->account->names->first()->name ?? '',
        'account_code' => $reportData->account->agent->code ?? $reportData->account->code ?? '',

      ];
    } catch (\Exception $e) {
      logger()->error('Failed to refresh totals: ' . $e->getMessage());
      $this->totals = [
        'opening_balance' => 0,
        'total_debit' => 0,
        'total_credit' => 0,
        'entry_count' => 0,
        'outTransfers' => 0,
        'inTransfers' => 0,
        'fromDate' => '',
        'toDate' => '',
        'currency' => 'TRY',
        'net_balance' => 0,
        'balance_type' => '',
        'currency_words' => '',
        'account_code' => '',
        'account_name' => '',
        'current_balance' => 0,
      ];
    }
  }


  public function render()
  {
    return view('livewire.statements.details');
  }
}
