<?php

namespace App\Livewire\Transfers;

use App\Events\RefreshDataTable;
use App\Models\Agent;
use App\Models\LedgerApproval;
use App\Models\LedgerBooking;
use App\Models\LedgerExchange;
use App\Models\LedgerTransfer;
use App\Services\Transfer\BookingService;
use App\Services\Transfer\TransferService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use OwenIt\Auditing\Events\AuditCustom;
use Illuminate\Support\Facades\Event;

class CancelTransfer extends Component
{
  public $transfer;
  protected function  getListeners()
  {
    return [
      'CancelTransfer' => 'CancelTransfer',
      'CancelTransferConfirm' => 'CancelTransferConfirm',
      'DeleteTransfer' => 'DeleteTransfer',
      'DeleteTransferConfirm' => 'DeleteTransferConfirm',
      'BlockTransferConfirm' => 'BlockTransferConfirm',
      'UnblockTransferConfirm' => 'UnblockTransferConfirm',
      'BlockTransfer' => 'BlockTransfer',
      'UnblockTransfer' => 'UnblockTransfer',
    ];
  }
  public function BlockTransferConfirm($transfer_id)
  {
    $this->dispatch('areyousure', ['msg' => __("You are about to block this transfer"), 'emitName' => 'BlockTransfer', 'action' => __("Yes"), 'title' => __("Block Transfer"), 'data' => ['transfer_id' => $transfer_id]]);
  }
  public function UnblockTransferConfirm($transfer_id)
  {
    $this->dispatch('areyousure', ['msg' =>  __("You are about to unblock this transfer"), 'emitName' => 'UnblockTransfer', 'action' => __("Yes"), 'title' => __("Unblock Transfer"), 'data' => ['transfer_id' => $transfer_id]]);
  }
  public function BlockTransfer($data)
  {
    try {
      DB::transaction(function () use ($data) {
        $transfer_id = $data['transfer_id'];
        $transfer = LedgerTransfer::find($transfer_id);
        BlockTransfer($transfer);
        $this->dispatch("sweetalert:success", [
          'msg' => __("alerts.transfer_blocked_successfully"),
          "title" => "Success",
          "type" => "success"
        ]);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'livewatch-table']);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'booking-table']);
        $this->dispatch("RefreshTransferInfo", ['transfer_id' => $transfer_id]);
      });
    } catch (\Exception $e) {
      Log::error('Error blocking transfer', ['error' => $e->getMessage()]);
      $this->dispatch("sweetalert:error", [
        'msg' => __("alerts.unexpected_error"),
        "title" => "Error",
        "type" => "error"
      ]);
    }
  }
  public function UnblockTransfer($data)
  {
    try {
      DB::transaction(function () use ($data) {
        $transfer_id = $data['transfer_id'];
        $transfer = LedgerTransfer::find($transfer_id);
        UnblockTransfer($transfer);
        $this->dispatch("sweetalert:success", [
          'msg' => __("alerts.transfer_unblocked_successfully"),
          "title" => "Success",
          "type" => "success"
        ]);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'livewatch-table']);
        $this->dispatch("RefreshYajraDatatable", ['table' => 'booking-table']);
        $this->dispatch("RefreshTransferInfo", ['transfer_id' => $transfer_id]);
      });
    } catch (\Exception $e) {
      Log::error('Error unblocking transfer', ['error' => $e->getMessage()]);
      $this->dispatch("sweetalert:error", [
        'msg' => __("alerts.unexpected_error"),
        "title" => "Error",
        "type" => "error"
      ]);
    }
  }
  public function  DeleteTransferConfirm($transfer_id)
  {
    $this->dispatch('areyousure', ['msg' => __("alerts.areyousure"), 'emitName' => 'DeleteTransfer', 'action' => __("actions.delete_transfer"), 'title' => __("actions.delete_transfer"), 'data' => ['transfer_id' => $transfer_id]]);
  }
  public function DeleteTransfer($data)
  {
    $transfer_id = $data['transfer_id'];

    try {
      DB::transaction(function () use ($transfer_id) {
        $this->transfer = LedgerTransfer::withoutGlobalScopes()->where('id', $transfer_id)->lockForUpdate()->first();

        if (!$this->transfer) {
          $this->dispatch("sweetalert:error", ['msg' => __("alerts.transfer_not_found"), "title" => "Error", "type" => "error"]);
          return;
        }
        $service = new TransferService();
        $service->DeleteLedger($this->transfer);
        $this->transfer->update(['status' => 'deleted']);
        $this->transfer->delete();
      });

      // fire outside transaction
      event(new RefreshDataTable("transfer-table"));
      event(new RefreshDataTable("livewatch-table"));

      $this->dispatch("sweetalert:success", [
        'msg' => __("alerts.transfer_deleted_successfully"),
        "title" => "Success",
        "type" => "success"
      ]);
    } catch (\Exception $e) {
      Log::error('Error deleting transfer', ['error' => $e->getMessage()]);
      $this->dispatch("sweetalert:error", [
        'msg' => __("alerts.unexpected_error"),
        "title" => "Error",
        "type" => "error"
      ]);
    }

    $this->dispatch("UnBlockUI");
  }
  public function  CancelTransferConfirm($transfer_id)
  {
    $this->dispatch('areyousure', ['msg' => __("alerts.areyousure"), 'emitName' => 'CancelTransfer', 'action' => __("actions.cancel_transfer"), 'title' => __("actions.cancel_transfer"), 'data' => ['transfer_id' => $transfer_id]]);
  }



  public function CancelTransfer($data)
  {
    $transfer_id = $data['transfer_id'];


    try {
      DB::transaction(function () use ($transfer_id) {

        $baseTransfer = LedgerTransfer::withoutGlobalScopes()->where('id', $transfer_id)->lockForUpdate()->first();
        $modelClass = match (strtolower($baseTransfer->type)) {
          'booking'   => LedgerBooking::class,
          'approval'  => LedgerApproval::class,
          'exchange'  => LedgerExchange::class,
          default     => LedgerTransfer::class,
        };


        // Hydrate into correct class
        $this->transfer = (new $modelClass)->newFromBuilder($baseTransfer->getAttributes(), $baseTransfer->getConnectionName());
        if (!$this->transfer) {
          $this->dispatch("sweetalert:error", ['msg' => __("alerts.transfer_already_cancelled"), "title" => "Error", "type" => "error"]);
          return;
        }

        if ($this->transfer->status === "cancelled") {
          $this->dispatch("sweetalert:error", ['msg' => __("alerts.transfer_already_cancelled"), "title" => "Error", "type" => "error"]);
          return;
        }

        $transfer_array = $this->transfer->toArray();
        $reference = generateReferenceNumber("DEL");
        $old_reference = $this->transfer->reference;
        $transfer_array['id'] = null;
        $transfer_array['status'] = "reverse";
        $transfer_array['reference'] = $reference;
        $transfer_array['type'] = "Booking";


        $transfer_array['delivery_amount'] = $this->transfer->amount;

        $transfer_array['delivery_currency'] = $this->transfer->currency;
        $transfer_array['amount'] = $this->transfer->delivery_amount;
        $transfer_array['currency'] = $this->transfer->delivery_currency;
        $transfer_array['currency'] = $this->transfer->delivery_currency;

        $transfer_array['old_reference'] = $old_reference;
        $transfer_array['transfer_id'] =  $this->transfer->id;
        if ($this->transfer->type == "Booking") {
          $transfer_array['sender_id'] = $this->transfer->receiver_id;
          $transfer_array['receiver_id'] = $this->transfer->sender_id;
        } else {
          $transfer_array['sender_id'] = $this->transfer->receiver_id; //==1300??Agent::where("code", 1300)->first()->id;
          $transfer_array['receiver_id'] = $this->transfer->sender_id;
        }
        $transfer_array['amount'] = $this->transfer->delivery_amount;
        $cloneTransfer = LedgerBooking::create($transfer_array);
        $service = new BookingService();
        $service->createTransferLedgerEntries($cloneTransfer);
        $old_status = $this->transfer->status;

        $this->transfer->disableAuditing();
        $this->transfer->status = "cancelled";
        //    $this->transfer->deleted_at = now();
        $this->transfer->save();
        $transfer = $this->transfer;
        $transfer::enableAuditing();

        $transfer->auditEvent = 'reverse_transfer';
        $transfer->isCustomEvent = true;
        $transfer->auditCustomOld = ["status" => $old_status, "reference" => $old_reference];
        $transfer->auditCustomNew = ["status" => "cancelled", "reference" => $reference];

        Event::dispatch(new AuditCustom($transfer));
      });

      // fire outside transaction
      event(new RefreshDataTable("transfer-table"));
      event(new RefreshDataTable("livewatch-table"));
      $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
      $this->dispatch("RefreshYajraDatatable", ['table' => 'livewatch-table']);
      $this->dispatch("RefreshYajraDatatable", ['table' => 'booking-table']);
      $this->dispatch("RefreshTransferInfo", ['transfer_id' => $transfer_id]);
      $this->dispatch("sweetalert:success", [
        'msg' => __("alerts.transfer_cancelled_successfully"),
        "title" => "Success",
        "type" => "success"
      ]);
    } catch (\Exception $e) {
      Log::error('Error cancelling transfer', ['error' => $e->getMessage()]);
      $this->dispatch("sweetalert:error", [
        'msg' => __("alerts.unexpected_error"),
        "title" => "Error",
        "type" => "error"
      ]);
    }

    $this->dispatch("UnBlockUI");
  }



  public function render()
  {
    return view('livewire.transfers.cancel-transfer');
  }
}
