<?php

namespace App\Livewire\TrialBalance;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Services\TrialBalanceService;
use Abivia\Ledger\Models\LedgerCurrency;
use Abivia\Ledger\Models\LedgerDomain;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class TrialBalanceIndex extends Component
{
  use WithFileUploads;

  // Form properties
  public $currency = '';
  public $depth = 3;
  public $toDate = '';
  public $domainUuid = '';
  public $includeZeroBalances = false;



  // Data properties
  public $trialBalanceData = [];
  public $summary = [];
  public $showResults = false;
  public $isLoading = false;
  public $errorMessage = '';

  // Available options
  public $currencies = [];
  public $domains = [];

  protected $trialBalanceService;

  public function mount()
  {
    $this->currencies = LedgerCurrency::orderBy('code')->get();
    $this->domains = LedgerDomain::get();

    // Set default date (only toDate is needed for Abivia API)
    $this->toDate = Carbon::now()->format('Y-m-d');

    // Initialize all properties to prevent hydration issues
    $this->trialBalanceData = [];
    $this->summary = [];
    $this->showResults = false;
    $this->isLoading = false;
    $this->errorMessage = '';
  }

  public function boot(TrialBalanceService $trialBalanceService)
  {
    $this->trialBalanceService = $trialBalanceService;
  }

  public function hydrate()
  {
    // Ensure all properties are properly initialized during hydration
    if (!is_array($this->trialBalanceData)) {
      $this->trialBalanceData = [];
    }
    if (!is_array($this->summary)) {
      $this->summary = [];
    }
    if (!is_bool($this->showResults)) {
      $this->showResults = false;
    }
    if (!is_bool($this->isLoading)) {
      $this->isLoading = false;
    }
    if (!is_string($this->errorMessage)) {
      $this->errorMessage = '';
    }
  }

  public function dehydrate()
  {
    // Ensure data is properly serialized during dehydration
    if (is_array($this->trialBalanceData)) {
      $this->trialBalanceData = array_values($this->trialBalanceData);
    }
  }

  private function clearComponentData()
  {
    $this->trialBalanceData = [];
    $this->summary = [];
    $this->errorMessage = '';
  }

  private function resetDataProperties()
  {
    $this->trialBalanceData = [];
    $this->summary = [];
    $this->errorMessage = '';
    $this->showResults = false;
  }

  private function processAndAssignData($data)
  {
    // Create a completely new array to avoid any reference issues
    $cleanData = [];
    $summaryData = [
      'total_balance' => 0,
      'total_debit' => 0,
      'total_credit' => 0,
      'total_accounts' => 0,
      'account_count' => 0,
      'is_balanced' => true,
      'balance_status' => 'balanced'
    ];

    foreach ($data as $account) {
      if (!is_array($account)) {
        continue;
      }

      // Create clean account data
      $cleanAccount = [
        'account_code' => (string)($account['account_code'] ?? ''),
        'account_name' => (string)($account['account_name'] ?? ''),
        'account_type' => (string)($account['account_type'] ?? ''),
        'depth' => (int)($account['depth'] ?? 1),
        'total_balance' => (float)($account['total_balance'] ?? 0),
        'debit_balance' => (float)($account['debit_balance'] ?? 0),
        'credit_balance' => (float)($account['credit_balance'] ?? 0),
        'opening_balance' => (float)($account['opening_balance'] ?? 0),
        'is_category' => (bool)($account['is_category'] ?? false),
        'ledger_uuid' => (string)($account['ledger_uuid'] ?? ''),
        'parent_code' => (string)($account['parent_code'] ?? ''),
        'movements' => [
          'transaction_count' => (int)($account['movements']['transaction_count'] ?? 0)
        ]
      ];

      $cleanData[] = $cleanAccount;

      // Calculate summary (exclude category accounts from balance calculations)
      $isCategory = $cleanAccount['is_category'];
      if (!$isCategory) {
        $summaryData['total_balance'] += $cleanAccount['total_balance'];
        $summaryData['total_debit'] += $cleanAccount['debit_balance'];
        $summaryData['total_credit'] += $cleanAccount['credit_balance'];
      }
      $summaryData['account_count']++;
      $summaryData['total_accounts']++;
    }

    // Round summary values
    $summaryData['total_balance'] = round($summaryData['total_balance'], 2);
    $summaryData['total_debit'] = round($summaryData['total_debit'], 2);
    $summaryData['total_credit'] = round($summaryData['total_credit'], 2);
    $summaryData['is_balanced'] = abs($summaryData['total_debit'] - $summaryData['total_credit']) <= 0.01;
    $summaryData['balance_status'] = $summaryData['is_balanced'] ? 'balanced' : 'unbalanced';

    // Assign data in a single operation
    $this->trialBalanceData = $cleanData;
    $this->summary = $summaryData;
  }

  public function generateTrialBalance()
  {
    $this->validate([
      'currency' => 'required|string',
      'depth' => 'required|integer|min:1|max:5',
      'toDate' => 'required|date',
      'domainUuid' => 'nullable|string',
    ]);

    try {
      $this->isLoading = true;
      $this->errorMessage = '';
      $this->showResults = false;

      // Clear previous data to prevent hydration issues
      $this->clearComponentData();

      // Special handling for USD currency
      if ($this->currency === 'USD') {
        Log::info('Generating USD trial balance with parameters:', [
          'currency' => $this->currency,
          'depth' => $this->depth,
          'toDate' => $this->toDate,
          'domainUuid' => $this->domainUuid,
          'includeZeroBalances' => $this->includeZeroBalances
        ]);
      }

      $data = $this->trialBalanceService->generateTrialBalance(
        $this->currency,
        $this->depth,
        null, // fromDate not needed for Abivia API
        $this->toDate,
        $this->domainUuid,
        $this->includeZeroBalances
      );

      // Additional validation for USD response
      if ($this->currency === 'USD' && !is_array($data)) {
        Log::error('USD trial balance service returned non-array data: ' . gettype($data));
        throw new \Exception('Invalid USD trial balance data structure');
      }

      // Ensure data is properly structured
      if (is_array($data)) {
        // Debug logging for USD currency
        if ($this->currency === 'USD') {
          Log::info('USD Trial Balance Data Structure:', [
            'count' => count($data),
            'first_item' => $data[0] ?? 'No data',
            'sample_keys' => array_keys($data[0] ?? [])
          ]);
        }

        // Process data directly without complex filtering
        $processedData = $this->processTrialBalanceData($data);

        // Debug logging
        Log::info('Trial Balance Processing Results:', [
          'currency' => $this->currency,
          'original_count' => count($data),
          'processed_count' => count($processedData)
        ]);

        // Process and assign data safely
        $this->processAndAssignData($processedData);
        $this->showResults = true;
      } else {
        throw new \Exception('Invalid data structure returned from service');
      }
    } catch (\Exception $e) {
      Log::error('Trial Balance Generation Error:', [
        'currency' => $this->currency,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      $this->errorMessage = 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage();
      $this->resetDataProperties();
    } finally {
      $this->isLoading = false;
    }
  }

  public function exportToExcel()
  {
    if (empty($this->trialBalanceData)) {
      $this->errorMessage = 'يرجى إنشاء التقرير أولاً قبل التصدير';
      return;
    }

    try {
      $filename = "trial_balance_{$this->currency}_to_{$this->toDate}.xlsx";

      return $this->trialBalanceService->exportToExcel(
        $this->trialBalanceData,
        $filename
      );
    } catch (\Exception $e) {
      $this->errorMessage = 'حدث خطأ أثناء التصدير: ' . $e->getMessage();
    }
  }

  public function resetForm()
  {
    $this->reset([
      'currency',
      'depth',
      'toDate',
      'domainUuid',
      'includeZeroBalances',

      'showResults',
      'trialBalanceData',
      'summary',
      'errorMessage'
    ]);

    // Reset to default date
    $this->toDate = Carbon::now()->format('Y-m-d');
    $this->isLoading = false;
  }



  private function processTrialBalanceData($data)
  {
    if (!is_array($data)) {
      return [];
    }

    $processed = [];

    foreach ($data as $account) {
      // Skip if not an array
      if (!is_array($account)) {
        continue;
      }

      // Always include all accounts in the display (both regular and category accounts)
      $isCategory = (bool)($account['is_category'] ?? false);

      // Create clean account data with all required fields
      $processedAccount = [
        'account_code' => (string)($account['account_code'] ?? ''),
        'account_name' => (string)($account['account_name'] ?? ''),
        'account_type' => (string)($account['account_type'] ?? ''),
        'depth' => (int)($account['depth'] ?? 1),
        'total_balance' => (float)($account['total_balance'] ?? 0),
        'debit_balance' => (float)($account['debit_balance'] ?? 0),
        'credit_balance' => (float)($account['credit_balance'] ?? 0),
        'opening_balance' => (float)($account['opening_balance'] ?? 0),
        'is_category' => $isCategory,
        'ledger_uuid' => (string)($account['ledger_uuid'] ?? ''),
        'parent_code' => (string)($account['parent_code'] ?? ''),
        'movements' => is_array($account['movements'] ?? null) ? $account['movements'] : ['transaction_count' => 0]
      ];

      $processed[] = $processedAccount;
    }

    return $processed;
  }

  private function ensureSerializableData($data)
  {
    if (!is_array($data)) {
      return [];
    }

    $serializable = [];

    foreach ($data as $index => $account) {
      if (!is_array($account)) {
        continue;
      }

      // Ensure all values are primitive types that Livewire can serialize
      $serializable[] = [
        'account_code' => (string)($account['account_code'] ?? ''),
        'account_name' => (string)($account['account_name'] ?? ''),
        'account_type' => (string)($account['account_type'] ?? ''),
        'depth' => (int)($account['depth'] ?? 1),
        'total_balance' => (float)($account['total_balance'] ?? 0),
        'debit_balance' => (float)($account['debit_balance'] ?? 0),
        'credit_balance' => (float)($account['credit_balance'] ?? 0),
        'opening_balance' => (float)($account['opening_balance'] ?? 0),
        'is_category' => (bool)($account['is_category'] ?? false),
        'ledger_uuid' => (string)($account['ledger_uuid'] ?? ''),
        'parent_code' => (string)($account['parent_code'] ?? ''),
        'movements' => [
          'transaction_count' => (int)($account['movements']['transaction_count'] ?? 0)
        ]
      ];
    }

    return $serializable;
  }

  private function calculateSummary($data)
  {
    $totalBalance = 0;
    $totalDebit = 0;
    $totalCredit = 0;
    $accountCount = 0;

    // Ensure data is an array and has the expected structure
    if (!is_array($data)) {
      return [
        'total_accounts' => 0,
        'total_balance' => 0,
        'total_debit' => 0,
        'total_credit' => 0,
        'is_balanced' => true,
        'balance_status' => 'balanced'
      ];
    }

    foreach ($data as $account) {
      if (is_array($account)) {
        // Exclude category accounts from balance calculations
        $isCategory = (bool)($account['is_category'] ?? false);

        if (!$isCategory) {
          // Only include regular accounts in balance calculations
          $totalBalance += (float)($account['total_balance'] ?? 0);
          $totalDebit += (float)($account['debit_balance'] ?? 0);
          $totalCredit += (float)($account['credit_balance'] ?? 0);
        }

        // Count all accounts (both regular and category) for display
        //  $accountCount++;
      }
    }

    // Round to 2 decimal places to avoid floating point precision issues
    $totalBalance = round($totalBalance, 2);
    $totalDebit = round($totalDebit, 2);
    $totalCredit = round($totalCredit, 2);

    return [
      'total_accounts' => $accountCount,
      'total_balance' => $totalBalance,
      'total_debit' => $totalDebit,
      'total_credit' => $totalCredit,
      'is_balanced' => abs($totalBalance) < 0.01,
      'balance_status' => $this->getBalanceStatus($totalBalance)
    ];
  }

  private function getBalanceStatus($totalBalance)
  {
    $absDifference = abs($totalBalance);
    if ($absDifference < 0.01) {
      return 'balanced';
    } elseif ($absDifference < 1.00) {
      return 'minor_imbalance';
    } else {
      return 'major_imbalance';
    }
  }

  public function getAccountTypeColor($type)
  {
    $colors = [
      'Asset' => 'primary',
      'Liability' => 'danger',
      'Equity' => 'success',
      'Revenue' => 'info',
      'Expense' => 'warning'
    ];

    return $colors[$type] ?? 'secondary';
  }

  public function getBalanceClass($balance)
  {
    if ($balance > 0) return 'balance-positive';
    if ($balance < 0) return 'balance-negative';
    return 'balance-zero';
  }

  public function render()
  {
    return view('livewire.trial-balance.trial-balance-index');
  }
}
