<?php

namespace App\Models;

use App\Interfaces\Avatar;
use App\Traits\HasAddress;
use App\Traits\HasAvatar;
use App\Traits\HasLedgerAccount;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\LedgerBalance;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class Agent extends Model  implements HasMedia, Avatar
{
  use HasFactory;
  // HasLedgerAccount;
  use HasAddress;
  use HasAvatar;
  use SoftDeletes;
  use InteractsWithMedia;

  protected $casts = [
    'is_active' => 'boolean',
    'created_at' => 'datetime:Y-m-d H:i:s',
    'updated_at' => 'datetime:Y-m-d H:i:s',
  ];

  /**
   * Boot the model and register event listeners
   */
  protected static function booted()
  {
    static::updating(function ($agent) {
      // Check if the name has changed
      if ($agent->isDirty('name')) {
        $oldName = $agent->getOriginal('name');
        $newName = $agent->name;

        // Update ledger names for this agent's accounts
        $agent->updateLedgerNames($oldName, $newName);
      }
    });
  }

  protected $fillable = [
    'name',
    'code',
    'phone',
    'email',
    'ledger_reference_uuid',
    'is_active',
    'credit_limit',
    'username',
    'password',
    'parent_id',
    'type',
    'is_main_cash',
  ];
  protected $type_colors = [
    "agent" => "primary",
    "cashier" => "success",
    "admin" => "danger",
    'branch' => 'warning',
    'account' => 'secondary',
    'Sender_receiver' => 'success',
    'Receiver' => 'warning',
    "Sender" => 'primary',
  ];

  public function getCodeAttribute()
  {
    if ($this->parent_id) {
      return $this->parent->code;
    }
    return $this->attributes['code'];
  }
  public function getDefaultAvatar()
  {
    $gender = $this->gender ?? "m";
    return '/assets/img/avatars/' . $gender . '.jpg';
  }
  public function getTypeHtml()
  {
    if ($this->type == "account")
      return '<span class="badge bg-secondary">' . __('agents.account') .  '</span>';
    $type =  "Sender";
    if ($this->GlobalCreditLimit->send_transfer ?? false)
      $type = __("Sender_receiver");
    if ($this->parent_id)
      $type = __("Receiver");
    return '<span class="badge bg-' . $this->type_colors[$type] . '">' . __('agents.' . $type) .  '</span>';
  }

  public function sentTransfers()
  {
    return $this->hasMany(LedgerTransfer::class, 'sender_id');
  }

  public function receivedTransfers()
  {
    return $this->hasMany(LedgerTransfer::class, 'receiver_id');
  }
  public  function undeliveredTransfers()
  {
    return $this->hasMany(LedgerTransfer::class, 'receiver_id')->where('status', 'pending');
  }
  public function getAllReceivedTransfers()
  {
    $ids = $this->rootAgent()->getallchildid();
    $ids[] = $this->id; // add self ID

    return LedgerTransfer::where("type", "transfer")->whereIn('receiver_id', $ids);
  }
  public  function users()
  {
    return $this->hasMany(User::class);
  }
  public function sypRates()
  {
    return $this->hasMany(AgentSypRate::class);
  }
  public function enabledSypRates()
  {
    return $this->hasMany(AgentSypRate::class)
      ->where('enabled', true)
      ->where('sell_price', '>', 0);
  }

  public function getBalanceAttribute()
  {
    if (!$this->ledger_uuid) {
      return 0;
    }

    $balance = \Abivia\Ledger\Models\LedgerBalance::where('ledgerUuid', $this->ledger_uuid)
      ->first();

    return $balance ? (float) $balance->balance : 0;
  }
  // In your Agent model
  public function hasNonZeroBalance(): bool
  {
    $real_agent_id = $this->rootAgent()->id;
    return $this->getAgentBasedBalances() != 0;
    // return LedgerBalance::query()
    //   ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
    //   ->where('ledger_accounts.agent_id', $real_agent_id )
    //   ->where(DB::raw('CAST(ledger_balances.balance AS DECIMAL(15,2))'), '!=', 0)
    //   ->exists();
  }
  public function getAgentBalances(): \Illuminate\Support\Collection
  {
    return LedgerBalance::query()
      ->select(
        'ledger_balances.currency',
        DB::raw('SUM(CAST(ledger_balances.balance AS DECIMAL(15,2))) as balance'),
        DB::raw('COUNT(DISTINCT ledger_balances.ledgerUuid) as account_count')
      )
      ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
      ->whereRaw("JSON_EXTRACT(ledger_accounts.extra, '$.agent_id') = ?", [$this->id])
      ->where("ledger_accounts.code", "like", "11%")
      ->groupBy('ledger_balances.currency')
      ->get()
      ->map(function ($item) {
        return [
          'currency' => $item->currency,
          'balance' => (float) $item->balance,
          'account_count' => $item->account_count,
          'formatted_balance' => PriceFormat(abs(round($item->balance)), $item->currency),
          'abs_balance' => abs($item->balance)
        ];
      });
  }
  public function  getAgentCurrencyBalance($currencyCode)
  {
    $balance = $this->getAgentBalances()->firstWhere('currency', $currencyCode);
    return $balance ? $balance['balance'] : 0;
  }

  public function getAgentBasedBalances()
  {
    $balances = $this->getAgentBalances();
    return  getBaseCurrencyBalance($balances);
  }
  public function getDetailedAgentBalances(): \Illuminate\Support\Collection
  {
    return LedgerBalance::query()
      ->select(
        'ledger_balances.currency',
        'ledger_balances.balance',
        'ledger_accounts.code as account_code',
        DB::raw('(SELECT name
                      FROM ledger_names
                      WHERE ownerUuid = ledger_accounts.ledgerUuid
                        AND language = "en"
                      LIMIT 1) as account_name')
      )
      ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
      ->whereRaw("JSON_EXTRACT(ledger_accounts.extra, '$.agent_id') = ?", [$this->id])
      ->orderBy('ledger_balances.currency')
      ->orderBy('ledger_accounts.code')
      ->get()
      ->groupBy('currency')
      ->map(function ($currencyGroup) {
        return [
          'total_balance' => $currencyGroup->sum(fn($item) => (float) $item->balance),
          'accounts' => $currencyGroup->map(function ($account) {
            return [
              'code' => $account->account_code,
              'name' => $account->account_name,
              'balance' => (float) round($account->balance),
              'formatted_balance' =>   number_format(round($account->balance, 2)) . ' ' . $account->currency
            ];
          })->values()
        ];
      });
  }

  // In your Agent model
  public static function getAgentsWithBalances(array $agentIds): Collection
  {
    return static::query()
      ->select('agents.*')
      ->selectSub(function ($query) {
        $query->selectRaw('SUM(CAST(lb.balance AS DECIMAL(15,2)))')
          ->from('ledger_balances as lb')
          ->join('ledger_accounts as la', 'lb.ledgerUuid', '=', 'la.ledgerUuid')
          ->whereColumn('la.extra->agent_id', 'agents.id')
          ->whereRaw('CAST(lb.balance AS DECIMAL(15,2)) != 0');
      }, 'has_balance')
      ->whereIn('id', $agentIds)
      ->get();
  }
  public function revenuepercent()
  {
    return $this->hasMany(RevenuePercent::class);
  }
  // In your Agent model
  public function getFormattedTotalBalance(): string
  {
    $total = $this->getAgentBalances()->sum('balance');
    $currency = $this->getAgentBalances()->first()['currency'] ?? '';

    return number_format(abs($total), 2) . ' ' . $currency .
      ($total < 0 ? ' (CR)' : ' (DR)');
  }
  public function getLedgerAccounts(): \Illuminate\Support\Collection
  {
    $accounts = LedgerAccount::with("names")->where('extra', 'like', '%"agent_id":' . $this->id . '%')
      ->get();

    return $accounts;
  }

  public  function generateAgentCode()
  {
    $prefix = "AGT_";
    $code = mt_rand(100000, 999999);
    $agent = Agent::where('code', $code)->first();
    if ($agent) {
      return $this->generateAgentCode();
    }
    return $prefix . $code;
  }
  public function ScopeActive($query)
  {
    return $query->where('is_active', 1);
  }
  public  function ScopeSubAgents($query)
  {
    return $query->whereNotNull('parent_id')->where("type", "agent");
  }
  public function ScopeIsAgent($query)
  {
    return $query->whereNull('parent_id')->where("type", "agent");
  }
  public function ScopeisActive($query)
  {
    return $query->where("is_active", 1);
  }
  public function ScopeIsCashier($query)
  {
    return $query->where("type", "cashier");
  }
  public function ScopeIsBranch($query)
  {
    return $query->whereNotNull('parent_id');
  }
  public function ScopeCashier($query)
  {
    return $query->where("type", "cashier");
  }
  public function ScopeAccount($query)
  {
    return $query->where("type", "account");
  }
  public function getCashBalance()
  {
    $accounts = $this->getAccounts();
    $cash = $accounts['cash'];

    return LedgerBalance::query()
      ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
      ->where('ledger_accounts.extra', 'like', '%"agent_id":' . $this->id . '%')
      ->where('ledger_accounts.code', $cash)
      ->first();
  }
  public  function getAccountBalance($code)
  {

    return LedgerBalance::query()
      ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
      ->where('ledger_accounts.extra', 'like', '%"agent_id":' . $this->id . '%')
      ->where('ledger_accounts.code', $code)
      ->first();
  }
  public function  getAccounts()
  {
    $account = $this;
    if ($this->parent_id) {
      $account = $this->parent;
    }

    return
      [
        'cash' => $account->getAgentAccountCode('11'),
        'receivable' => $account->getAgentAccountCode('12'), //المستحقات من العميل
        'payable' => $account->getAgentAccountCode('21'), //المستحقات للعميل
        'not_delivered' => "1300",
        'revenue' => $account->getAgentAccountCode('41'), //الارباح للشركة
        'unearned_revenue' => "5100",
        'commission' => $account->getAgentAccountCode('51') // نضع فيها  الارباح  للعميل  كمصروفات
      ];
  }
  public function getMainCashierAgent()
  {
    return $this->children()->where('is_main_cash', 1)->first();
  }
  public  function  CreateMainCashierAgent()
  {
    $agent = [
      'parent_id' => $this->id,
      'code' => $this->generateAgentCode(),
      'name' => "MainCashier_" . "$this->name.",
      'is_main_cash' => 1,
      'is_active' => 1,
      'type' => 'cashier',

    ];
    return Agent::create($agent);
  }
  public function getMainCashWindow()
  {
    return $this->getAgentAccountCode('MAINCASH_11');
  }
  public function getAgentCashAccount()
  {
    return $this->getAccounts()['cash'];
  }
  public function getAgentAccountCode(string $prefix): string
  {
    $account = LedgerAccount::where('extra->agent_id', $this->id)
      ->where('code', 'like', $prefix . '%')
      ->first();

    return $account->code ?? "";
  }
  public  function  ledger_accounts()
  {
    return $this->hasMany(CustomLedgerAccount::class,  'agent_id', 'id');
  }
  public function parent()
  {
    return $this->belongsTo(Agent::class, 'parent_id');
  }

  public function children()
  {
    return $this->hasMany(Agent::class, 'parent_id');
  }
  public function rootAgent()
  {
    return $this->parent ?? $this;
  }
  public function transferPackages()
  {
    return $this->belongsToMany(TransferPackage::class, 'agents_transfer_packages', 'agent_id', 'package_id');
  }
  public function transferPackageOverrides()
  {
    return $this->hasMany(AgentTransferPackageDetail::class);
  }
  public  function CurrencyTransferPackages($currency)
  {
    return TransferPackage::where("type", 'transfer')->where('currency', $currency)->where(function ($q) {
      $q->whereHas('agents', fn($q) => $q->where('agent_id', $this->id))->orwherenull('agent_id');
    })->with('details')->first(); //$this->belongsToMany(TransferPackage::class, 'agents_transfer_packages', 'agent_id', 'package_id')->where('currency', $currency)->orwhereNull("transfer_packages.agent_id")->first();
  }
  public  function CurrencyApprovalPackages($currency)
  {
    return TransferPackage::where("type", 'approval')->where('currency', $currency)->where(function ($q) {
      $q->whereHas('agents', fn($q) => $q->where('agent_id', $this->id))->orwherenull('agent_id');
    })->with('details')->first(); //$this->belongsToMany(TransferPackage::class, 'agents_transfer_packages', 'agent_id', 'package_id')->where('currency', $currency)->orwhereNull("transfer_packages.agent_id")->first();
  }
  public  function getAgentCashBalance()
  {
    $accounts = $this->getAccounts();
    $cash = $accounts['cash'];

    $balances = BalanceOverwrite::wherehas("account", function ($query) use ($cash) {
      return  $query->where("code", $cash);
    })->get();
    return  $balances;
  }

  public  function  CheckBalance($currency, $amount)
  {
    $balances = $this->getAgentCashBalance();
    $balance = $balances->where("currency", $currency)->first();
    if ($balance) {
      if ($balance->balance >= $amount) {
        return true;
      }
    }
    return false;
  }
  // App\Models\Agent.php

  public function getFeeForPackage(int $packageId, float $amount): ?array
  {
    // 1. Check agent-specific override
    $override = $this->transferPackageOverrides()
      ->where('package_id', $packageId)
      ->where('min_amount', '<=', $amount)
      ->where('max_amount', '>=', $amount)
      ->first();

    if ($override) {
      return [
        'send_fee' => $override->send_fee,
        'receive_fee' => $override->receive_fee,
        'send_fee_type' => $override->send_fee_type,
        'receive_fee_type' => $override->receive_fee_type,
        'source' => 'agent override',
      ];
    }

    // 2. Default package detail fallback
    $package = TransferPackage::with('details')->find($packageId);

    if (!$package) return null;

    $defaultDetail = $package->details
      ->where('min_amount', '<=', $amount)
      ->where('max_amount', '>=', $amount)
      ->first();

    if ($defaultDetail) {
      return [
        'send_fee' => $defaultDetail->send_fee,
        'receive_fee' => $defaultDetail->receive_fee,
        'send_fee_type' => $defaultDetail->send_fee_type,
        'receive_fee_type' => $defaultDetail->receive_fee_type,
        'source' => 'default',
      ];
    }

    return null;
  }


  public function getallchildid()
  {
    $children = $this->rootAgent()->children()->pluck('id')->toArray();
    array_push($children, $this->id);
    return $children;
  }
  public function isitChild($id)
  {
    if (in_array($id, $this->getallchildid())) {
      return true;
    }
    return false;
  }
  public function currencyVisibilities()
  {
    return $this->hasMany(\App\Models\AgentCurrencyVisibility::class);
  }
  public function canSendIn($currencyCode): bool
  {
    return $this->currencyVisibilities()
      ->where('currency_code', $currencyCode)
      ->where('can_send', true)
      ->exists();
  }

  public function canReceiveIn($currencyCode): bool
  {
    return $this->currencyVisibilities()
      ->where('currency_code', $currencyCode)
      ->where('can_receive', true)
      ->exists();
  }

  public function GlobalCreditLimit()
  {
    return $this->hasOne(AgentGlobalCreditLimit::class);
  }

  public function CurrencyCreditLimit()
  {
    return $this->hasMany(AgentCurrencyCreditLimit::class);
  }
  public function TransferGlobalCreditLimit()
  {
    return $this->hasOne(AgentGlobalCreditLimit::class);
  }

  public function getTransferGlobalCreditLimit()
  {
    return $this->TransferGlobalCreditLimit;
  }
  public function getCanSendCurrency()
  {
    return $this->currencyVisibilities()
      ->where('can_send', true)
      ->with('currency')
      ->get()
      ->pluck('currency')   // get only the related Currency models
      ->filter()             // remove nulls (in case relation missing)
      ->values();            // reindex keys
  }
  public function getCanReceiveCurrency()
  {
    return $this->currencyVisibilities()
      ->where('can_receive', true)
      ->with('currency')
      ->get()
      ->pluck('currency')   // get only the related Currency models
      ->filter()             // remove nulls (in case relation missing)
      ->values();            // reindex keys
  }
  public  function GlobalAccreditationCreditLimit()
  {
    return $this->hasOne(AgentGlobalCreditLimit::class);
  }

  public function getGlobalAccreditationCreditLimit()
  {
    return $this->GlobalAccreditationCreditLimit;
  }
  public function ApiConfig()
  {
    return $this->hasOne(ApiConfig::class);
  }

  /**
   * Get all ledger accounts for this agent
   */
  public function ledgerAccounts()
  {
    return $this->hasMany(\Abivia\Ledger\Models\LedgerAccount::class, 'agent_id');
  }

  /**
   * Get all ledger accounts including those in extra JSON field
   */
  public function getAllLedgerAccounts()
  {
    return \Abivia\Ledger\Models\LedgerAccount::where('agent_id', $this->id)
      ->orWhereJsonContains('extra->agent_id', $this->id)
      ->get();
  }

  /**
   * Preview what ledger names would be changed
   */
  public function previewLedgerNameChanges($oldName, $newName)
  {
    try {
      if (empty($oldName) || empty($newName)) {
        return [];
      }

      $ledgerAccounts = $this->getAllLedgerAccounts();
      $preview = [];

      foreach ($ledgerAccounts as $ledgerAccount) {
        $ledgerNames = DB::table('ledger_names')
          ->where('ownerUuid', $ledgerAccount->ledgerUuid)
          ->get(['language', 'name']);

        foreach ($ledgerNames as $ledgerName) {
          if (str_contains($ledgerName->name, $oldName)) {
            $newNamePreview = str_replace($oldName, $newName, $ledgerName->name);
            $preview[] = [
              'ledgerUuid' => $ledgerAccount->ledgerUuid,
              'code' => $ledgerAccount->code,
              'language' => $ledgerName->language,
              'old_name' => $ledgerName->name,
              'new_name' => $newNamePreview
            ];
          }
        }
      }

      return $preview;
    } catch (\Exception $e) {
      Log::error("Failed to preview ledger name changes: " . $e->getMessage());
      return [];
    }
  }

  /**
   * Manually update ledger names for this agent
   */
  public function updateLedgerNames($oldName, $newName)
  {
    try {
      // Safety checks
      if (empty($oldName) || empty($newName)) {
        Log::warning("Cannot update ledger names: old name or new name is empty");
        return 0;
      }

      if ($oldName === $newName) {
        Log::info("No ledger name update needed: names are identical");
        return 0;
      }

      // Get all ledger accounts for this agent (including JSON extra field)
      $ledgerAccounts = $this->getAllLedgerAccounts();

      if ($ledgerAccounts->isEmpty()) {
        Log::info("No ledger accounts found for agent '{$oldName}'");
        return 0;
      }

      $updatedCount = 0;

      foreach ($ledgerAccounts as $ledgerAccount) {
        // Update ledger names in both languages
        $result = DB::table('ledger_names')
          ->where('ownerUuid', $ledgerAccount->ledgerUuid)
          ->update([
            'name' => DB::raw("REPLACE(name, '{$oldName}', '{$newName}')")
          ]);

        if ($result > 0) {
          $updatedCount += $result;
        }
      }

      Log::info("Updated {$updatedCount} ledger names for agent '{$oldName}' to '{$newName}'");

      return $updatedCount;
    } catch (\Exception $e) {
      Log::error("Failed to update ledger names for agent '{$oldName}': " . $e->getMessage());
      throw $e; // Re-throw to let the calling code handle the error
    }
  }

  /**
   * Manually update ledger names for this agent
   */
  public function refreshLedgerNames()
  {
    $oldName = $this->getOriginal('name') ?? $this->name;
    return $this->updateLedgerNames($oldName, $this->name);
  }

  /**
   * Get a summary of ledger accounts and names for this agent
   */
  public function getLedgerAccountsSummary()
  {
    try {
      $ledgerAccounts = $this->getAllLedgerAccounts();
      $summary = [];

      foreach ($ledgerAccounts as $ledgerAccount) {
        $ledgerNames = DB::table('ledger_names')
          ->where('ownerUuid', $ledgerAccount->ledgerUuid)
          ->get(['language', 'name']);

        $summary[] = [
          'code' => $ledgerAccount->code,
          'ledgerUuid' => $ledgerAccount->ledgerUuid,
          'names' => $ledgerNames->toArray()
        ];
      }

      return $summary;
    } catch (\Exception $e) {
      Log::error("Failed to get ledger accounts summary: " . $e->getMessage());
      return [];
    }
  }

  /**
   * Test the ledger name update functionality
   */
  public function testLedgerNameUpdate($testName = 'TestAgentName')
  {
    try {
      $currentName = $this->name;

      // Preview the changes
      $preview = $this->previewLedgerNameChanges($currentName, $testName);

      if (empty($preview)) {
        return [
          'success' => false,
          'message' => 'No changes would be made'
        ];
      }

      // Make the test change
      $updatedCount = $this->updateLedgerNames($currentName, $testName);

      // Revert the change
      $revertedCount = $this->updateLedgerNames($testName, $currentName);

      return [
        'success' => true,
        'preview' => $preview,
        'updated_count' => $updatedCount,
        'reverted_count' => $revertedCount,
        'message' => "Successfully tested: {$updatedCount} names updated and {$revertedCount} reverted"
      ];
    } catch (\Exception $e) {
      return [
        'success' => false,
        'message' => 'Test failed: ' . $e->getMessage()
      ];
    }
  }
}
