<?php

namespace App\Models;

use Abivia\Ledger\Models\JournalEntry;
use Abivia\Ledger\Models\LedgerAccount;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Rawilk\Settings\Facades\Settings;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Support\Arr;
use App\Traits\CustomAuditable;
use App\Traits\TransfersTrait;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use OwenIt\Auditing\Contracts\Auditable;

class LedgerApproval extends Model implements HasMedia, Auditable
{
  protected $table = 'ledger_transfers';
  use SoftDeletes;
  use TransfersTrait;
  use HasFactory;
  use InteractsWithMedia;
  use \OwenIt\Auditing\Auditable, CustomAuditable;
  protected static function booted()
  {
    static::addGlobalScope('approval', function ($query) {
      $query->where('type', 'approval');
    });

    static::addGlobalScope('agentTransfers', function (Builder $builder) {
      if (Auth::guard('agent')->check() && Auth::guard("agent")->user()->agent_id != null) {

        $agentId = Auth::guard('agent')->user()->agent->rootAgent()->id;

        $builder->where(function ($query) use ($agentId) {
          $query->where('sender_id', $agentId)
            ->orWhere('receiver_id', $agentId);
        });
      }
    });
  }




  protected $types = [
    'transfer' => ['template' => 'vertical'],
    'data' => [['sender_id' => 'select'], 'receiver_id', 'amount', 'currency', 'receiver_fee', 'sender_fee', 'fee', 'status', 'reference'],
    'record' => ['sender_id', 'reciever_id']
  ];

  protected $filters = [
    'sender_id',
    'receiver_id',
    'type',
    'status',
    'created_at',
    'delivered_at',
    'currency',
    'approval_type',
    'from_date',
    'to_date',

  ];
  protected $fillable = [
    'id',
    'receiver_fee',

    'sender_id',
    'receiver_id',
    'amount',
    'delivery_amount',
    'fee',
    'currency',
    'status',
    'reference',
    'ledger_journal_uuid',
    'customer_details',
    'id',
    'debit_journal_uuid',
    'credit_journal_uuid',
    'client_id',
    'send_fee',
    'delivery_currency',
    'notes',
    'secret',
    'type',
    'date',
    'recipient_name',
    'recipient_client_id',
    'recipient_phone',
    'recipient_id_number',
    'client_sender_name',
    'client_sender_phone',
    'client_sender_id_number',
    'sender_account_code',
    'receiver_account_code',
    'user_id',
    'rate',
    'rate_factor',
    'external_ref',
    'transfer_id'



  ];
  protected $appends = ['status_color'];
  protected $statusColors = [
    'pending' => 'warning',
    'approved' => 'success',
    'rejected' => 'danger',
    'cancelled' => 'danger',
    'completed' => 'success',
    'deleted' => 'danger',
    'reversed' => 'danger'
  ];

  protected $casts = [
    'customer_details' => 'array',
    'extra' => 'array',

    'created_at' => 'datetime:Y-m-d H:i:s',
    'delivered_at' => 'datetime:Y-m-d H:i:s'
  ];
  public function reason()
  {
    return $this->belongsTo(Reason::class);
  }
  public function sender()
  {
    return $this->belongsTo(Agent::class, 'sender_id', 'id');
  }

  public function receiver()
  {
    return $this->belongsTo(Agent::class, 'receiver_id', 'id');
  }
  public function debitJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'debit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public function creditJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'credit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public  function getStatusColorAttribute()
  {
    return $this->statusColors[$this->status];
  }
  public  function  client()
  {
    return $this->belongsTo(Client::class);
  }



  public static function getFilterableFields()
  {

    return collect([
      __('transfers.sender_agent') => ["key" => "sender_id", 'type' => 'x-Mulk.select2', 'options' =>  self::GetSourceAgents()->pluck('name', 'id')->toArray()],
      __('transfers.receiver_agent') => ["key" => "receiver_id", 'type' => 'x-Mulk.select2', 'options' => self::GetDestinationAgents()->pluck('name', 'id')->toArray()],
      __('transfers.status') => ["key" => "status", 'type' => 'x-Mulk.select2', 'options' => ['pending' => __("transfers.pending"), 'completed' => __("transfers.completed")]],
      __("transfers.from_date") => [
        'key' => 'from_date',
        'type' => 'date',
        'options' => []
      ],
      __("transfers.to_date") => [
        'key' => 'to_date',
        'type' => 'date',
        'options' => []
      ],
      __("transfers.delivered_at") => [
        'key' => 'delivered_at',
        'type' => 'date-range',
        'options' => [
          'start' => null,
          'end' => null
        ]
      ],

    ])
      ->filter(fn($config) => is_array($config) && isset($config['type']))
      ->toArray();
  }
  public function getData($queryBuilder, array $filters = [], $type = "data", $datatype = "all")
  {


    foreach ($filters as $filter => $value) {
      if (!in_array($filter, $this->filters)) {
        continue;
      }
      if ($filter === 'from_date' && $value) {
        $queryBuilder->whereDate('created_at', ">=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'to_date' && $value) {
        $queryBuilder->whereDate('created_at', "<=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'delivered_at' && is_array($value) && count($value) === 2) {
        $value["start"] = Carbon::parse($value["start"])->format("Y-m-d");
        $value["end"] = Carbon::parse($value["end"])->format("Y-m-d");
        $queryBuilder->whereDate($filter, ">=", $value["start"])->whereDate($filter, "<=", $value["end"]);
      } else {
        if (isset($value) && $value != "") {

          $queryBuilder = $this->getquery($queryBuilder, $filter, $value);
        }
      }
    }
    if ($type == "data") {

      return $queryBuilder;
    } else
      return $queryBuilder->count();
  }
  public  function getquery($query, $key, $value)
  {

    if ($key == "approval_type") {
      if ($value == "external") {
        return $query->External();
      }
      if ($value == "internal") {
        return $query->Internal();
      }
    }

    $query = $query->where($key, $value);

    return $query;
  }
  public function scopeFilter($query, $filters)
  {
    $query = $this->getData($query, $filters);

    return $query;
  }
  public function ScopeApprovalTransfer($query)
  {
    return $query->where('type', 'approval');
  }


  public function getLedgerJournal()
  {
    return $this->belongsTo(JournalEntry::class, 'ledger_journal_uuid', 'journalEntryId');
  }
  public function user()
  {
    return $this->belongsTo(User::class);
  }
  public  function getParsedReceipt()
  {
    try {
      $type = $this->type;
      $template_id = settings::get("default_template.$type");
      $template = Template::find($template_id);
      $html = $template->content;
      $companyName = Settings::get('general.company_name');
      $logo =  '/storage/Logo/' . Settings::get('logo', null);
      return str_replace([
        '%company_name%',
        '%logo%',
        '%date%',
        '%receiver_id%',
        '%sender_id%',
        '%reference%',
        '%client_name%',
        '%client_phone%',
        '%secret%',
        '%identity_number%',
        '%notes%',
        '%amount%',
        '%send_fee%',
        '%receive_fee%',
        '%delivered_at%',
        '%receiver_address%',
        '%sender_address%',



      ], [
        $companyName, // Or get from DB
        "<img src=\"$logo\" alt=\"Logo\" class=\"img-fluid mt-2 embed-responsive\" style=\"max-width: 100px; max-height: 100px;\" />",
        $this->created_at->format('Y-m-d H:i:s'),
        $this->receiver->name ?? '',
        $this->sender->name ?? '',
        $this->reference,
        $this->client->name,
        $this->client->phone,
        $this->secret,
        $this->identity_number,
        $this->notes,
        format_money($this->amount, "USD"),
        format_money($this->send_fee, "USD"),
        format_money($this->receive_fee, "USD"),
        $this->delivered_at,
        $this->receiver->getFullAddress() ?? '',
        $this->sender->getFullAddress() ?? '',


      ], $html);
    } catch (\Exception $e) {
      return "الرجاء تعيين  قالب  لاشعار الحوالة ";
    }
  }
  public function transformAudit(array $data): array
  {
    if (Arr::has($data, 'new_values.sender_id')) {

      $data['new_values']['sender_id'] =  $this->sender->name;
    }
    if (Arr::has($data, 'new_values.receiver_id')) {

      $data['new_values']['sender_id'] =  $this->receiver->name;
    }
    if (Arr::has($data, 'new_values.amount')) {

      $data['new_values']['amount'] = format_money($this->amount, $this->currency);
    }
    if (Arr::has($data, 'new_values.delivery_amount')) {

      $data['new_values']['delivery_amount'] = format_money($this->delivery_amount, $this->delivery_currency);
    }




    return $data;
  }
  public static function getDeliveryRules(): array
  {
    return [
      'transfer.reference' => 'required',

      'secret' => 'required',
    ];
  }
  public  function ScopeExternal($query)
  {

    return $query->whereNotNull('external_ref');
  }
  public  function  ScopeInternal($query)
  {

    return $query->whereNull('external_ref');
  }
  public  static function GetSourceAgents()
  {
    if (auth()->guard('agent')->check()) {
      $agent = auth()->guard('agent')->user()->agent->rootAgent();
      return Agent::IsAgent()->where('id', $agent->id)->get();
    }
    return Agent::IsAgent()->get();
  }

  public  static function GetDestinationAgents()
  {

    $all_child_ids = [];
    if (Auth::guard('agent')->check()) {
      $agent = Auth::guard('agent')->user()->agent;
      $all_child_ids = $agent->getallchildid();
    }

    return  Agent::whereHas("GlobalCreditLimit", function ($query) {
      $query->where('send_approval', true);
    })->whereNotIn('id', $all_child_ids)
      ->IsAgent()->where('type', '!=', 'cashier')->get();
  }
  public  function DeliveredCurrency()
  {
    return $this->belongsTo(Currency::class, 'delivery_currency', 'code');
  }

  public function SenderCurrency()
  {
    return $this->belongsTo(Currency::class, 'currency', 'code');
  }
  public function journalEntries()
  {
    return JournalEntry::where('extra->exchange_id', $this->id)->get();
  }
  public function  isExternalTransfer()
  {
    return $this->api_id && $this->external_agent_id;
  }
  public function isReversed()
  {
    return JournalEntry::where('extra->reverses_transfer_id', $this->id)->exists() || $this->status == "cancelled";
  }
  public  function isDelivered()
  {
    return $this->delivered_at != null || $this->status == "completed";
  }
}
