<?php

namespace App\Models;

use Abivia\Ledger\Models\JournalEntry;
use Abivia\Ledger\Models\LedgerAccount;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Rawilk\Settings\Facades\Settings;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Support\Arr;
use App\Traits\CustomAuditable;
use App\Traits\HasAddress;
use App\Traits\TransfersTrait;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Laravel\Scout\Searchable;
use OwenIt\Auditing\Contracts\Auditable;

class LedgerTransfer extends Model implements HasMedia, Auditable
{
  use HasFactory;
  use InteractsWithMedia;
  use \OwenIt\Auditing\Auditable, CustomAuditable;
  use SoftDeletes;
  use Searchable;
  use HasAddress;
  use TransfersTrait;

  public $meilisearchSettings = [
    'searchableAttributes' => [
      'reference',
      'sender_name',
      'receiver_name',
      'recipient_name',
      'recipient_phone',
      'recipient_id_number',
      'sender_phone',
      'sender_id_number',
      'receiver_phone',
      'receiver_id_number',
      'notes',
      'receiver_note',
    ],
    'filterableAttributes' => [
      '__soft_deleted',
      'status',
      'currency',
      'sender_id',
      'receiver_id',
      'type',
      'recipient_name',
      'recipient_phone',
      'amount',
      'delivery_amount',
      'send_fee',
      'receiver_fee',
      'created_at',
      'delivered_at',
    ],
    'sortableAttributes' => [
      'amount',
      'delivery_amount',
      'send_fee',
      'receiver_fee',
      'created_at',
      'delivered_at',
    ],
    'displayedAttributes' => ['*'], // show all fields
    'rankingRules' => [
      'words',
      'typo',
      'proximity',
      'attribute',
      'exactness',
    ],
  ];



  public function toSearchableArray()
  {
    return [
      'id' => $this->id,
      'reference' => $this->reference,
      'sender_name' => $this->sender->name ?? "",
      'receiver_name' => $this->receiver->name ?? "",
      'amount' => $this->amount,
      'currency' => $this->currency,
      'status' => $this->status,
      'recipient_phone' => $this->recipient_phone,
      'recipient_id_number' => $this->recipient_id_number,
      "recipient_name" => $this->recipient_name,

      "sender_phone" => $this->sender->phone ?? "",
      "sender_id_number" => $this->sender->id_number ?? "",
      "sender_account_code" => $this->sender->account_code ?? "",

      "receiver_phone" => $this->receiver->phone ?? "",
      "receiver_id_number" => $this->receiver->id_number ?? "",
      "receiver_account_code" => $this->receiver->account_code ?? "",
      'notes' => $this->notes,
      'receiver_note' => $this->receiver_note,
      'deleted_at' => $this->deleted_at,
      '__soft_deleted' => $this->deleted_at != null,
    ];
  }

  /**
   * Determine if the model should be searchable.
   * Include soft-deleted records in search results.
   */
  public function shouldBeSearchable()
  {
    // Include both active and soft-deleted records in search
    return true;
  }
  protected $types = [
    'transfer' => ['template' => 'vertical'],
    'data' => [['sender_id' => 'select'], 'receiver_id', 'amount', 'currency', 'receiver_fee', 'sender_fee', 'fee', 'status', 'reference'],
    'record' => ['sender_id', 'reciever_id']
  ];

  protected $search_filters = [];
  protected $filters = [
    'sender_id',
    'receiver_id',
    'type',
    'status',
    'created_at',
    'delivered_at',
    'currency',
    'show',
    'recipient_name',
    'recipient_phone',
    'client_sender_name',
    'client_sender_phone',
    'amount',
    'delivery_currency',
    'delivery_amount',
    'send_fee',
    'receiver_fee',
    'from_date',
    'to_date',
    'fee_type',

  ];
  protected $fillable = [
    'id',
    'receiver_fee',

    'sender_id',
    'receiver_id',
    'amount',
    'delivery_amount',
    'fee',
    'currency',
    'status',
    'reference',
    'ledger_journal_uuid',
    'customer_details',
    'id',
    'debit_journal_uuid',
    'credit_journal_uuid',
    'client_id',
    'send_fee',
    'delivery_currency',
    'notes',
    'secret',
    'type',
    'date',
    'user_id',
    'recipient_name',
    'recipient_client_id',
    'recipient_phone',
    'recipient_id_number',
    'client_sender_name',
    'client_sender_phone',
    'client_sender_id_number',
    'sender_account_code',
    'receiver_account_code',
    'rate',
    'rate_factor',
    'external_ref',
    'api_id',
    'external_agent_id',
    'reason_id',
    'is_paid',
    'is_collected',
    'transfer_id',
    'fee_type'




  ];
  protected $appends = ['status_color', 'reason_name'];
  protected $statusColors = [
    'pending' => 'warning',
    'approved' => 'success',
    'rejected' => 'danger',
    'cancelled' => 'danger',
    'completed' => 'success',
    'reverse' => 'danger',
    'deleted' => 'danger',
    'blocked' => 'danger',
    'booking' => 'info',
  ];

  protected $casts = [
    'customer_details' => 'array',

    'extra' => 'array',
    'created_at' => 'datetime:Y-m-d H:i:s',
    'delivered_at' => 'datetime:Y-m-d H:i:s'
  ];

  protected static function booted()
  {
    static::addGlobalScope('agentTransfers', function (Builder $builder) {
      if (Auth::guard('agent')->check() && Auth::guard("agent")->user()->agent_id != null) {

        $agentId = Auth::guard('agent')->user()->agent->rootAgent()->id;

        $builder->where(function ($query) use ($agentId) {
          $query->where('sender_id', $agentId)
            ->orWhere('receiver_id', $agentId);
        });
      }
    });
    static::creating(function ($transfer) {
      if ($transfer->sender instanceof Agent) {
        $transfer->sender_id = $transfer->sender->rootAgent()->id;
      }
    });
  }
  public  function getStatusAttribute($value)
  {
    if ($this->type == "booking")
      return $value == "completed" ? 'booking' : $value;
    return $value;
  }
  public function reason()
  {
    return $this->belongsTo(Reason::class);
  }
  public function sender()
  {
    return $this->belongsTo(Agent::class);
  }

  public function receiver()
  {
    if ($this->type == "approval")
      return $this->belongsTo(LedgerAccount::class, 'receiver_account_code', 'code');
    return $this->belongsTo(Agent::class);
  }
  public function debitJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'debit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public function creditJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'credit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public  function getStatusColorAttribute()
  {
    return $this->statusColors[$this->status];
  }
  public  function  client()
  {
    return $this->belongsTo(Client::class);
  }
  public  function sender_client()
  {
    return $this->belongsTo(Client::class, 'client_id');
  }
  public function  recipient_client()
  {
    return $this->belongsTo(Client::class, 'recipient_client_id');
  }
  public function isReversed()
  {
    return  $this->status == "cancelled";
  }
  public  function isDelivered()
  {
    return $this->delivered_at != null || $this->status == "completed";
  }

  public  function DeliveredCurrency()
  {
    return $this->belongsTo(Currency::class, 'delivery_currency', 'code');
  }

  public function SenderCurrency()
  {
    return $this->belongsTo(Currency::class, 'currency', 'code');
  }

  public static function getFilterableFields()
  {
    if (
      auth()->guard('agent')->check()
      && auth()->guard('agent')->user()->agent_id != null
    ) {
      return collect([

        __("transfers.from_date") => [
          'key' => 'from_date',
          'type' => 'date',
          'options' => []
        ],
        __("transfers.to_date") => [
          'key' => 'to_date',
          'type' => 'date',
          'options' => []
        ],

      ])->filter(fn($config) => is_array($config) && isset($config['type']))
        ->toArray();
    }
    return collect([
      __('transfers.sender_agent') => ["key" => "sender_id", 'type' => 'x-Mulk.select2', 'options' =>  Agent::pluck('name', 'id')->toArray()],
      __('transfers.receiver_agent') => ["key" => "receiver_id", 'type' => 'x-Mulk.select2', 'options' => Agent::pluck('name', 'id')->toArray()],
      __('transfers.status') => ["key" => "status", 'type' => 'x-Mulk.select2', 'options' => ['' => __("transfers.all"), 'pending' => __("transfers.pending"), 'completed' => __("transfers.completed")]],

      __("transfers.from_date") => [
        'key' => 'from_date',
        'type' => 'date',
        'options' => []
      ],
      __("transfers.to_date") => [
        'key' => 'to_date',
        'type' => 'date',
        'options' => []
      ],

    ])
      ->filter(fn($config) => is_array($config) && isset($config['type']))
      ->toArray();
  }
  public function getData($queryBuilder, array $filters = [], $type = "data", $datatype = "all")
  {
    Log::info("filters: " . json_encode($filters));
    $this->search_filters = $filters;
    foreach ($filters as $filter => $value) {

      if (!in_array($filter, $this->filters)) {

        continue;
      }
      if ($filter === 'from_date' && $value) {
        $queryBuilder->whereDate('created_at', ">=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'to_date' && $value) {
        $queryBuilder->whereDate('created_at', "<=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'delivered_at' && is_array($value) && count($value) === 2) {
        $value["start"] = Carbon::parse($value["start"])->format("Y-m-d");
        $value["end"] = Carbon::parse($value["end"])->format("Y-m-d");
        $queryBuilder->whereDate($filter, ">=", $value["start"])->whereDate($filter, "<=", $value["end"]);
      } else {
        if (isset($value) && $value != "") {

          $queryBuilder = $this->getquery($queryBuilder, $filter, $value);
        }
      }
    }
    if ($type == "data") {

      return $queryBuilder;
    } else
      return $queryBuilder->count();
  }
  public function getquery($query, $key, $value)
  {
    Log::info("Filtering by: $key => $value");

    if ($value == null || $value == "" || $value == "all") {
      return $query;
    }
    // Handle show filter
    if ($key == 'show') {
      Log::info($value);
      if ($value == 'all') {
        return $query;
      }
      return $query->where('status', $value);
    }



    // Handle recipient name search
    if ($key == "recipient_name") {
      return $query->where(function ($q) use ($value) {
        $q->where("customer_details->name->ar", 'LIKE', "%$value%")
          ->orWhere("customer_details->name->en", 'LIKE', "%$value%");
      });
    }

    // Handle recipient phone search
    if ($key == "recipient_phone") {
      return $query->where(function ($q) use ($value) {
        $q->where("customer_details->phone", $value)
          ->orWhere("customer_details->mobile", $value);
      });
    }
    Log::info($this->search_filters);

    // Handle amount fields with operator (default to '=' if not specified)
    if (in_array($key, ['amount', 'delivery_amount', 'send_fee', 'receiver_fee'])) {
      $operator = $this->search_filters[$key . '_operator'] ?? '=';
      Log::info("Operator: $operator");
      return $query->where($key, $operator, $value);
    }
    if ($key == "status" && $value == "deleted") {
      return $query->onlyTrashed();
    }

    // Default case for other filters
    return $query->where($key, $value);
  }


  public function scopeFilter($query, $filters)
  {
    $query = $this->getData($query, $filters);

    return $query;
  }
  public function ScopeBookingTransfer($query)
  {
    return $query->where('type', 'booking');
  }
  public function ScopeTransfer($query)
  {
    return $query->whereIn('type', ['transfer', 'syp']);
  }
  public function ScopeExchange($query)
  {
    return $query->where('type', 'exchange');
  }

  public function getLedgerJournal()
  {
    return $this->belongsTo(JournalEntry::class, 'ledger_journal_uuid', 'journalEntryId');
  }
  public function user()
  {
    return $this->belongsTo(User::class);
  }
  public  function getParsedReceipt($direction = "send")
  {
    try {
      $type = $this->type;

      $template_id = Settings::get("default_template.transfer");
      if ($direction != "send") {
        $template_id = Settings::get("default_template.delivery");
      }

      $template = Template::find($template_id);

      $html = $template->content;

      $companyName = Settings::get('general.company_name');
      $logo =  '/storage/Logo/' . Settings::get('logo', null);
      $delivery_amount = $this->delivery_amount;
      if ($type == "syp") {
        $delivery_amount =  $this->syp_amount;
        $this->delivery_currency = "SYP";
      }
      return str_replace([
        '%company_name%',
        '%logo%',
        '%date%',
        '%receiver_id%',
        '%sender_id%',
        '%reference%',
        '%client_name%',
        '%client_phone%',
        '%secret%',
        '%identity_number%',
        '%notes%',
        '%amount%',
        '%delivery_amount%',
        '%send_fee%',
        '%receive_fee%',
        '%delivered_at%',
        '%receiver_address%',
        '%sender_address%',



      ], [
        $companyName, // Or get from DB
        "<img src=\"$logo\" alt=\"Logo\" class=\"img-fluid mt-2 embed-responsive\" style=\"max-width: 100px; max-height: 100px;\" />",
        $this->created_at->format('Y-m-d H:i:s'),
        $this->receiver->name ?? '',
        $this->sender->name ?? '',
        $this->reference,
        $this->client->name,
        $this->client->phone,
        $this->secret,
        $this->identity_number,
        $this->notes,
        format_money($this->amount, $this->currency),
        format_money($delivery_amount, $this->delivery_currency),
        format_money($this->send_fee, $this->currency),
        format_money($this->receive_fee, $this->delivery_currency),
        $this->delivered_at,
        $this->getFullAddress() ?? $this->receiver->getFullAddress() ?? '',
        $this->sender->getFullAddress() ?? '',


      ], $html);
    } catch (\Exception $e) {
      Log::error("Error generating transfer receipt: " . $e);
      return "الرجاء تعيين  قالب  لاشعار الحوالة ";
    }
  }
  public function transformAudit(array $data): array
  {
    if (Arr::has($data, 'new_values.sender_id')) {

      $data['new_values']['sender_id'] =  $this->sender->name;
    }
    if (Arr::has($data, 'new_values.receiver_id')) {

      $data['new_values']['sender_id'] =  $this->receiver->name;
    }
    if (Arr::has($data, 'new_values.amount')) {

      $data['new_values']['amount'] = format_money($this->amount, $this->currency);
    }
    if (Arr::has($data, 'new_values.delivery_amount')) {

      $data['new_values']['delivery_amount'] = format_money($this->delivery_amount, $this->delivery_currency);
    }




    return $data;
  }
  public static function getDeliveryRules(): array
  {
    return
      $rules = [
        'transfer.reference' => 'required',
        'transfer.recipient_client_id' => 'required',
        'transfer.recipient_id_number' => 'required',
        'recipient.identity_number' => 'required',
        'transfer.secret' => 'required',
      ];
  }
  public  static function GetSourceAgents()
  {

    return Agent::IsAgent()->get();
  }
  public static function GetDestinationAgents($type = "sending")
  {
    $all_child_ids = [];
    if (Auth::guard('agent')->check()) {
      $agent = Auth::guard('agent')->user()->agent;
      $all_child_ids = $agent->getallchildid();
    }



    // Internal agents
    $internalAgents = Agent::whereHas("GlobalCreditLimit", function ($query) {
      $query->where('send_transfer', true);
    })
      ->where("type", '!=', 'account')->where("type", '!=', 'cashier')
      ->whereNotIn('id', $all_child_ids)

      ->get();

    $externalAgents = ExternalAgent::all();

    $merged = $internalAgents->merge($externalAgents);

    // Then map *once* at the very end
    return $merged->map(function ($agent) {
      if ($agent instanceof Agent) {
        return [
          'id'       => 'internal_' . $agent->id,
          'text'     => $agent->name . " - " . $agent->code . " - " . ($agent->address?->country ?? "") . "-" . ($agent->address?->city ?? ""),
          'status'   => 'active',
          'source'   => 'internal',
          'agent_id' => $agent->id
        ];
      }
      return [
        'id'       => $agent->source_id . "_" . $agent->external_id,
        'text'     => $agent->name . " - " . $agent->code . " - " . ($agent->address?->country ?? "") . "-" . ($agent->address?->city ?? ""),
        'status'   => 'active',
        'source'   => 'external',
        'agent_id' => $agent->external_id
      ];
    })->sortBy('text', SORT_NATURAL | SORT_FLAG_CASE)->values();
  }

  public function  isExternalTransfer()
  {
    return $this->api_id && $this->external_agent_id;
  }
  public function journalEntries()
  {
    return $this->hasMany(JournalEntry::class, 'transfer_id', 'id');
  }
  public function getReasonNameAttribute()
  {
    return $this->reason?->name;
  }
}
