<?php

namespace App\Services\Ledger;


use Abivia\Ledger\Http\Controllers\LedgerAccountController;
use Abivia\Ledger\Messages\Account;
use Abivia\Ledger\Messages\Create;
use App\Models\Agent;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Abivia\Ledger\Exceptions\Breaker;
use App\Models\Currency;
use Database\Seeders\AgentSeeder;
use Database\Seeders\UserSeeder;

class AgentLedgerService
{
  function generateAgentAccountCode(string $baseCode, int $agentId): string
  {


    $prefix = substr($baseCode, 0, 4); // "11" for "1100"
    $agentSuffix = str_pad($agentId, 7, '0', STR_PAD_LEFT); // "0000001" for 1, "0000002" for 2

    return $prefix . $agentSuffix;
  }

  public function createLedgerCashAccountsForAgent(Agent $agent)
  {
    Log::info("Creating ledger accounts for agent: {$agent->name} (ID: {$agent->id})");
    try {
      $controller = new LedgerAccountController();

      if (!$agent->ledger_reference_uuid) {
        $agent->ledger_reference_uuid = (string) Str::uuid();
        $agent->save();
      }

      if ($agent->type == "agent") {




        $accounts = [

          [
            'code' =>  generateAgentAccountCode('5100', $agent->id),
            'names' => [
              ['language' => 'en', 'name' => $agent->name . 'Earned Commissions'],
              ['language' => 'ar', 'name' => 'عمولات  محققة ' . $agent->name]
            ],
            'parent' => '5100',
            'debit' => true
          ],


        ];
      }

      foreach ($accounts as $account) {
        $response = $controller->add(Account::fromArray(
          array_merge($account, [
            'reference' => $agent->ledger_reference_uuid,
            'extra' => json_encode(['agent_id' => $agent->id, 'type' => $account['type'] ?? $agent->type])
          ])
        ));
        $response->agent_id = $agent->id;

        $response->save();
      }

      // Create currency cash accounts
      // $this->CreateCurrencyCashLedger($agent);
      return true;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error('Agent account creation failed: ' . $e->getMessage());
      // return false;
    }
  }



  public function createLedgerAccountsForAgent(Agent $agent)
  {
    Log::info("Creating ledger accounts for agent: {$agent->name} (ID: {$agent->id})");
    try {
      $controller = new LedgerAccountController();

      if (!$agent->ledger_reference_uuid) {
        $agent->ledger_reference_uuid = (string) Str::uuid();
        $agent->save();
      }

      if ($agent->type == "agent") {




        $accounts = [
          // Cash Account (under Company Cash)
          [
            'code' =>  generateAgentAccountCode('1100', $agent->id),
            'names' => [
              ['language' => 'en', 'name' => $agent->name . ' Cash'],
              ['language' => 'ar', 'name' => 'نقدية ' . $agent->name]
            ],
            'parent' => '1100',
            'debit' => true,
            'category' => false,
          ],


          // Realized Fees (shared account, no agent-specific sub-accounts)

          // Agent Commissions
          [
            'code' =>  generateAgentAccountCode('1200', $agent->id),
            'names' => [
              ['language' => 'en', 'name' => $agent->name . 'unwithdrawn commissions'],
              ['language' => 'ar', 'name' => 'عمولات  غير مسحوبة للفروع ' . $agent->name]
            ],
            'parent' => '1200',
            'debit' => true
          ],


        ];
      } else {
        $accounts = [
          [
            'code' => generateAgentAccountCode('1100', $agent->id),
            'names' => [
              ['language' => 'en', 'name' => $agent->name . ' Cash'],
              ['language' => 'ar', 'name' => 'نقدية ' . $agent->name]
            ],
            'parent' => '1100',
            'debit' => true
          ],

        ];
      }

      foreach ($accounts as $account) {
        $response = $controller->add(Account::fromArray(
          array_merge($account, [
            'reference' => $agent->ledger_reference_uuid,
            'extra' => json_encode(['agent_id' => $agent->id, 'type' => $account['type'] ?? $agent->type])
          ])
        ));
        $response->agent_id = $agent->id;

        $response->save();
      }

      // Create currency cash accounts
      // $this->CreateCurrencyCashLedger($agent);
      return true;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error('Agent account creation failed: ' . $e->getMessage());
      // return false;
    }
  }
  public function CreateCurrencyCashLedger($agent)
  {
    $controller = new LedgerAccountController();
    $currencies = Currency::get();
    foreach ($currencies as $currency) {
      $currencyAccountCode = generateAgentCurrencyCashCode($agent->id, $currency->id); // e.g. 110101

      $controller->add(Account::fromArray([
        'code' => $currencyAccountCode,
        'names' => [
          ['language' => 'en', 'name' => $agent->code . "_" . $currency->code . ' Cash'],
          ['language' => 'ar', 'name' => 'نقدية ' . $agent->code . "_" . $currency->code]
        ],
        'parent' => $this->generateAgentAccountCode('1100', $agent->id), // Agent cash
        'debit' => true,
        'category' => false,
        'reference' => $agent->ledger_reference_uuid,
        'extra' => json_encode([
          'agent_id' => $agent->id,
          'currency_code' => $currency->code,
          'currency_id' => $currency->id,
        ])
      ]));
    }
  }
  public function CreateCurrencyExchangeSellBuyLedger($currency)
  {
    try {
      $controller = new LedgerAccountController();
      $exchange_code = $currency->GetExchangeAccountCode()['exchange'];
      // $sell_account_code = $currency->GetExchangeAccountCode()['sell'];
      // $buy_account_code = $currency->GetExchangeAccountCode()['buy'];


      $accounts = [
        // Cash Account (under Company Cash)
        [
          'code' =>  $exchange_code,
          'names' => [
            ['language' => 'en', 'name' => $currency->code . ' Exchange'],
            ['language' => 'ar', 'name' => 'مركز قطع ' . $currency->name]

          ],
          'parent' => '1400',
          'debit' => true,
          'category' => false,
        ],
        // Sell Account
        // [
        //     'code' =>  $sell_account_code,
        //     'names' => [
        //         ['language' => 'en', 'name' => $currency->code . ' Sell'],
        //         ['language' => 'ar', 'name' => 'بيع عملات ' . $currency->code]
        //     ],
        //     'parent' => $exchange_code,
        //     'debit' => true,
        //     'category' => false,
        // ],
        // // Buy Account
        // [
        //     'code' =>  $buy_account_code,
        //     'names' => [
        //         ['language' => 'en', 'name' => $currency->code . ' Buy'],
        //         ['language' => 'ar', 'name' => 'شراء عملات ' . $currency->code]
        //     ],
        //     'parent' => $exchange_code,
        //     'debit' => true,
        //     'category' => false,
        // ],
      ];



      foreach ($accounts as $account) {
        $response = $controller->add(Account::fromArray(
          array_merge($account, [
            'reference' => $currency->code,
            'extra' => json_encode(['currency_id' => $currency->id, 'currency_code' => $currency->code])
          ])
        ));
      }

      return true;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
      //   return false;
    } catch (\Exception $e) {
      Log::error('Agent account creation failed: ' . $e->getMessage());
      // return false;
    }
  }
  public function createLedgerAccountsForAllAgents(): void
  {
    $agents = Agent::get();

    foreach ($agents as $agent) {

      if ($this->createLedgerAccountsForAgent($agent)) {
        echo "Agent {$agent->name} seeded and ledger accounts created.\n";
        // $this->createCurrencyCashLedger($agent);
      } else {
        echo "Error creating ledger accounts for agent {$agent->name}.\n";
      }
    }
  }
  public  function createNewCurrencyCashLedger($currency)
  {
    try {
      $agents = Agent::get();
      foreach ($agents as $agent) {
        $controller = new LedgerAccountController();
        $currencyAccountCode = generateAgentCurrencyCashCode($agent->id, $currency->id); // e.g. 110101

        $response = $controller->add(Account::fromArray([
          'code' => $currencyAccountCode,
          'names' => [
            ['language' => 'en', 'name' => $agent->code . "_" . $currency->code . ' Cash'],
            ['language' => 'ar', 'name' => 'نقدية ' . $agent->code . "_" . $currency->code]
          ],
          'decimals' => 3,
          'parent' => $this->generateAgentAccountCode('1100', $agent->id), // Agent cash
          'debit' => true,
          'category' => false,
          'reference' => $agent->ledger_reference_uuid,
          'extra' => json_encode([
            'agent_id' => $agent->id,
            'currency_code' => $currency->code,
            'currency_id' => $currency->id,
          ])
        ]));
      }

      return true;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    }
  }




  public static function createAgentLedgerFromTemplate(string $templateName = 'sections'): bool
  {
    try {
      $controller = new LedgerAccountController();

      $createMessage = Create::fromArray([
        'names' => [
          ['lang' => 'en', 'name' => 'FastTransfer Inc.'],
        ],
        'currencies' => [
          [
            'code' => 'USD',
            'name' => 'US Dollar',
            'decimals' => 10
          ]
        ],
        'template' => $templateName,
        'domains' => [
          [
            'code' => 'default',
            'name' =>  'Default Domain'
          ]
        ],
        'rules' => [
          'account' => [
            'postToCategory' => false,
          ],
          'language' => [
            'default' => 'en',
          ],
          'openDate' => now()->format('Y-m-d'),
        ],
        'transDate' => now()->format('Y-m-d'), // Added opening balance date
      ]);

      $response = $controller->create($createMessage);


      return true;
    } catch (Breaker $e) {
      Log::info($e);
      return false;
      // dd([
      //     'exception_code' => $e->getCode(),
      //     'breaker_errors' => $e->getErrors(true), // Force all hidden errors
      //     'validation_rules' => config('ledger.rules'), // Check config
      // ]);
    } catch (\Exception $e) {
      Log::info($e);
      return false;
      // dd([
      //     'exception_code' => $e->getCode(),
      //     'breaker_errors' => $e->getErrors(true), // Force all hidden errors
      //     'validation_rules' => config('ledger.rules'), // Check config
      // ]);
    }
  }
  public function  Setup()
  {

    $this->createAgentLedgerFromTemplate();
    $this->createLedgerAccountsForAllAgents();
  }
}
