<?php

namespace App\Services;

use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Messages\Account;
use Abivia\Ledger\Messages\Domain;
use Abivia\Ledger\Messages\Name;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\LedgerDomain;
use RuntimeException;

class LedgerSetupService
{
    public function initializeLedger(): void
    {
        $this->resetLedger();
        // $this->createLedgerDomain();
        $this->createBaseAccounts();
    }

    protected function resetLedger(): void
    {
        LedgerAccount::resetRules();
    }

    protected function createLedgerDomain(): void
    {
        // Create the name object first
        $name = new Name();
        $name->name = 'Main Domain';
        $name->language = 'en';

        // Create the domain message
        $domain = new Domain();
        $domain->code = 'main'; // Will be auto-converted to 'MAIN'
        $domain->currencyDefault = 'USD';
        $domain->names = [$name];
        $domain->extra = json_encode(['description' => 'Primary operating domain']);

        try {
            $controller = new \Abivia\Ledger\Http\Controllers\LedgerDomainController();
            $response = $controller->run($domain, Domain::OP_ADD);

            if (!$response->success) {
                throw new RuntimeException('Failed to create ledger domain: controller returned failure');
            }

            // Verify the domain was actually created
            if (!LedgerDomain::where('code', 'MAIN')->exists()) {
                throw new RuntimeException('Domain record not found after creation');
            }
        } catch (Breaker $e) {
            throw new RuntimeException(
                "Domain creation failed: " . $e->getMessage(),
                $e->getCode(),
                $e
            );
        }
    }

    protected function createBaseAccounts(): void
    {
        $this->createRootAccount();
        $this->createAccount('1', 'Assets', '0', true, false);
        $this->createAccount('1100', 'Agent Funds', '1', true, false);
        $this->createAccount('2', 'Liabilities', '0', false, true);
        $this->createAccount('2100', 'Agent Obligations', '2', false, true);
        $this->createAccount('4', 'Income', '0', false, true);
        $this->createAccount('4100', 'Transfer Fees', '4', false, true);
    }

    protected function createRootAccount(): void
    {
        $name = new Name();
        $name->name = 'Root Account';
        $name->language = 'en';

        $account = new Account();
        $account->code = '0';
        $account->names = [$name];
        $account->debit = false;
        $account->credit = false;

        $this->runAccountOperation($account, Account::OP_ADD);
    }

    protected function createAccount(
        string $code,
        string $name,
        string $parentCode,
        bool $debit,
        bool $credit
    ): void {
        $nameObj = new Name();
        $nameObj->name = $name;
        $nameObj->language = 'en';

        $account = new Account();
        $account->code = $code;
        $account->parent = $parentCode;
        $account->names = [$nameObj];
        $account->debit = $debit;
        $account->credit = $credit;

        $this->runAccountOperation($account, Account::OP_ADD);
    }

    protected function runAccountOperation(Account $account, string $operation): void
    {
        try {
            $controller = new \Abivia\Ledger\Http\Controllers\LedgerAccountController();
            $response = $controller->run($account, $operation);

            if (!$response->success) {
                throw new RuntimeException("Failed to create account {$account->code}");
            }
        } catch (Breaker $e) {
            throw new RuntimeException(
                "Account operation failed: " . $e->getMessage(),
                $e->getCode(),
                $e
            );
        }
    }

    public function verifySetup(): bool
    {
        return LedgerDomain::where('code', 'MAIN')->exists()
            && LedgerAccount::where('code', '0')->exists()
            && LedgerAccount::where('code', '1100')->exists()
            && LedgerAccount::where('code', '2100')->exists()
            && LedgerAccount::where('code', '4100')->exists();
    }
}
