<?php


namespace App\Services\Partners;

use Illuminate\Support\Facades\Log;
use App\Interfaces\PartnerApiInterface;
use App\Models\Agent;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use phpseclib3\Crypt\RSA;
use phpseclib3\Math\BigInteger;
use SimpleXMLElement;

class BestRateApi implements PartnerApiInterface
{
    protected string $token;
    protected array $config;
    protected string $baseUrl;

    public function setConfig(array $config): void
    {
        $this->config = $config;
        $this->baseUrl = rtrim($config['base_url'], '/');

        // Force HTTPS if not already set properly
        if (!str_starts_with($this->baseUrl, 'https://')) {
            $this->baseUrl = preg_replace('/^http:\/\//', 'https://', $this->baseUrl);
        }
    }

    public function authenticate(): bool
    {
        $credentials = [
            'account'   => $this->config['Account'],
            'branch_id' => $this->config['BranchId'],
            'username'  => $this->config['UserName'],
            'password'  => $this->config['Password'],
            'key'       => $this->config['public_key'],
        ];

        try {
            $keyParts = $this->parseRsaXml($credentials['key']);
            $credentials['encrypted_password'] = $this->encryptPassword($credentials['password'], $keyParts);

            $authXml = $this->generateAuthXml($credentials);
            $authToken = base64_encode($authXml);

            $this->token = "PEF1dGhlbnRpY2F0aW9uPjxBY2NvdW50PjE0MTAxMDAwMjAwMTwvQWNjb3VudD48QnJhbmNoSWQ+MjA8L0JyYW5jaElkPjxVc2VyTmFtZT5mcmVlYnVya2FuPC9Vc2VyTmFtZT48UGFzc3dvcmQ+UWxaTDV5YVJocU4wSnoyUUJrSVYzTnRWMk1RRTN5bGk4c3VDRmhnbW93SlQ5amFzNUtMRG1iWkRaVjZKR3RDSk9sKzNsVzZmY1U1UitVRkJjZm9XTXc9PTwvUGFzc3dvcmQ+PC9BdXRoZW50aWNhdGlvbj4="; //$authToken;

            //   dd("{$this->baseUrl}/Adjustment/SearchAccounts");
            $response = Http::withHeaders([
                'Authorization' => 'Token ' . $this->token,

            ])->get("{$this->baseUrl}/Adjustment/SearchAccounts");


            dd($response->json());
            return true;
        } catch (\Exception $e) {
            Log::error('Auth Exception', ['error' => $e->getMessage()]);
        }

        return false;
    }

    public function getClientList(string $name, string $currency): array
    {
        if (!$this->authenticate()) {
            return [];
        }

        $response = Http::withHeaders([
            'Content-Type'  => 'application/json',
            'Authorization' => 'Token ' . $this->token,
        ])->get("{$this->baseUrl}/Adjustment/SearchAccounts", [
            'SearchStr'    => $name,
            'CurrencyCode' => $currency,
        ]);

        if (!$response->ok()) {
            Log::error('Client List Fetch Failed', [
                'status' => $response->status(),
                'body'   => $response->body(),
            ]);
            return [];
        }

        return collect($response->json()['AdditionalData'] ?? [])->map(fn($item) => [
            'id'     => $item['Key'],
            'text'   => $item['Value'],
            'status' => 'active',
        ])->all();
    }

    public function sendTransaction(array $data): array
    {
        if (!$this->authenticate()) {
            return [];
        }

        $payload = [
            'Amount'          => $data['amount'],
            'CurrencyCode'    => $data['currency'],
            'ToAccountID'     => $data['receiver_id'],
            'ToAccountName'   => $data['receiver_name'] ?? 'Unknown',
            'FromAccountInfo' => Agent::find($data['sender_id'])->name ?? 'Unknown Sender',
            'RefNumber'       => $data['reference'],
            'Note'            => $data['notes'] ?? '',
        ];

        $response = Http::withHeaders([
            'Content-Type'  => 'application/json',
            'Authorization' => 'Token ' . $this->token,
        ])->post("{$this->baseUrl}/Adjustment/SendTransaction", $payload);

        return $this->handleApiResponse($response);
    }

    public  function getAllClients()
    {
        return [];
    }
    protected function handleApiResponse($response): array
    {
        if (!$response->ok()) {
            Log::error('BestRate API Error', [
                'status' => $response->status(),
                'body'   => $response->body(),
            ]);

            return [
                'success' => false,
                'error'   => 'HTTP_' . $response->status(),
                'message' => 'API request failed.',
            ];
        }

        $body = $response->json();

        return match ($body['message'] ?? '') {
            '101'     => ['success' => false, 'error' => 'DUPLICATE_TRANSACTION', 'message' => 'Duplicate transaction.'],
            '102'     => ['success' => false, 'error' => 'INSUFFICIENT_BALANCE', 'message' => 'Insufficient balance.'],
            '01020'   => ['success' => false, 'error' => 'PERMISSION_OR_CURRENCY_ERROR', 'message' => 'Permission or currency mismatch.'],
            'bad request' => ['success' => false, 'error' => 'BAD_REQUEST', 'message' => 'Invalid request.'],
            ''        => ['success' => false, 'error' => 'EMPTY_RESPONSE', 'message' => 'No response message.'],
            default   => ['success' => false, 'error' => 'UNKNOWN_ERROR', 'message' => $body['message']],
        };
    }

    public function testConnection(): bool
    {
        return $this->authenticate();
    }

    public function encryptPassword(string $password, array $keyParts): string
    {
        $modulus = new BigInteger(base64_decode($keyParts['modulus']), 256);
        $exponent = new BigInteger(base64_decode($keyParts['exponent']), 256);

        $rsa = RSA::loadPublicKey(['n' => $modulus, 'e' => $exponent])
            ->withPadding(RSA::ENCRYPTION_PKCS1);

        return base64_encode($rsa->encrypt($password));
    }

    public function generateAuthXml(array $credentials): string
    {
        $xml = new SimpleXMLElement('<Authentication/>');
        $xml->addChild('Account', $credentials['account']);
        $xml->addChild('BranchId', $credentials['branch_id']);
        $xml->addChild('UserName', $credentials['username']);
        $xml->addChild('Password', $credentials['encrypted_password']);

        return $xml->asXML();
    }

    public function parseRsaXml(string $xml): array
    {
        $xmlObj = simplexml_load_string($xml);

        if (!$xmlObj || empty($xmlObj->Modulus) || empty($xmlObj->Exponent)) {
            throw new \InvalidArgumentException("Invalid RSA key XML format");
        }

        return [
            'modulus'  => (string) $xmlObj->Modulus,
            'exponent' => (string) $xmlObj->Exponent,
        ];
    }
}
