<?php
// app/Services/Partners/OrangeferApiService.php
namespace App\Services\Partners;

use Illuminate\Support\Facades\Log;
use FontLib\Table\Type\name;
use App\Interfaces\PartnerApiInterface;
use App\Models\Agent;
use App\Models\LedgerApproval;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;

class OrangeferApi implements PartnerApiInterface
{
    protected string $token;
    protected array $config;
    protected string $baseUrl;

    public function setConfig(array $config): void
    {
        $this->config = $config;
        $this->baseUrl = $config['base_url'];
    }



    public function authenticate(): bool
    {
        // Try to get token from cache
        $cachedToken = Cache::get('orangefer_api_token');

        if ($cachedToken) {
            $this->token = $cachedToken;
            return true;
        }

        // Token not cached, perform authentication
        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'username' => $this->config['username'],
            'password' => $this->config['password'],
            'agentcode' => $this->config['agentcode'],
        ])->post("{$this->baseUrl}/hmz191/api1d/login");

        Log::info('Orangefer Auth Response: ' . $response->body());

        if ($response->ok()) {
            $data = $response->json();

            if (isset($data['token'])) {
                $this->token = $data['token'];

                // Cache token for 55 minutes or appropriate expiry
                Cache::put('orangefer_api_token', $this->token, now()->addMinutes(55));

                return true;
            } else {
                Log::warning('Token missing from auth response', $data);
            }
        } else {
            Log::error('Orangefer auth failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
        }

        return false;
    }

    ///////////////////////////Approval///////////////////////////////////
    public function getClientList(string $name, string $currency): array
    {
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/namequery", [
            'name' => $name,
            'currency' => $currency,
        ]);

        $clients = $response->json() ?? [];

        return collect($clients)->map(function ($client) {
            return [
                'id' => $client['CID'],
                'text' => $client['CID'] . '-' . $client['CN'],
                'status' => 'active',
            ];
        })->all();
    }

    public function sendApproval(array $data): array
    {

        $ref =  $data['reference'];
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }
        $data['sender_name'] = Agent::where('id', $data['sender_id'])->first()->name;
        $array = [];
        $array['ACID'] = $data['receiver_id'];
        $array['SenderName'] = $data['sender_name'];

        $array['ANote'] = $data['notes'];
        $array['ASAM'] = $data['amount'];
        $array['ASCur'] = $data['currency'];

        $array['EXTNum'] = $data['reference'];

        $result = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/approval", $array);
        return $this->handleApiResponse($result);
    }
    //////////////////////////////Transaction/////////////////////////
    public function getAllClients()
    {
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/gettarget", []);

        $clients = $response->json() ?? [];

        Log::info('Orange AllClients Response: ' . $response->body());
        return  collect($clients)->map(function ($client) {
            return [
                'id' => $client['id'],
                'text' => $client['name'] . ' - ' . $client['id'] . '-' . $client['country'] ?? "",
                'status' => 'active',
                'source' => 'external',
                'address' => $client['address'] . "-" . $client['country'] . "-" . $client['city'] . "-" . $client['phone'],
            ];
        })->all();
        return $clients;
    }
    public  function getClientData($id)
    {

        $clients = $this->getAllClients();

        $client = collect($clients)->firstWhere('id', $id);


        return $client ?? [];
    }
    public function sendTransaction(array $data): array
    {

        $ref =  $data['reference'];
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }
        $data['sender_name'] = Agent::where('id', $data['sender_id'])->first()->name;
        $array = [];
        $array['target'] = $data['receiver_id'];
        $array['rcvname'] = $data['recipient_name'] ?? "";

        $array['rcvphone'] = $data['recipient_phone'] ?? "";
        $array['currency'] = strtolower($data['delivery_currency']);
        $array['amount'] = $data['delivery_amount'];

        $array['sender'] = $data['sender_name'] ?? "";
        $array['externalnum'] = $data['reference'];

        $result = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/newtrans", $array);
        return $this->handleApiResponse($result);
    }
    public function UpdateTransaction($transaction)
    {
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }

        $data = [
            'transnum' => $transaction->external_ref,
            'rcvname' => $transaction->recipient_name ?? "",
            'rcvphone' => $transaction->recipient_phone ?? "",
            'notes' => $transaction->notes ?? "",
            'sender' => $transaction->sender->name,
        ];


        $result = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/update", $data);
        return $this->handleSimpleResponse($result, 'updated');
    }

    public function DeleteTransaction($transaction)
    {
        $data = [
            'transnum' => $transaction->external_ref,

        ];

        $result = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/hmz191/api1d/delete", $data);
        return $this->handleSimpleResponse($result, 'deleted');
    }
    protected function handleSimpleResponse($response, string $action): array
    {
        if (!$response->ok()) {
            Log::error("Orangefer API {$action} error", [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return [
                'success' => false,
                'error' => 'HTTP ' . $response->status(),
                'message' => 'Request failed.',
            ];
        }

        $body = $response->json();
        $message = strtolower(trim($body['message'] ?? ''));

        if ($message === 'ok') {
            return [
                'success' => true,
                'message' => "Transaction successfully {$action}."
            ];
        }

        return [
            'success' => false,
            'error' => 'API_ERROR',
            'message' => "Failed to {$action} transaction: " . ($body['message'] ?? 'Unknown error'),
        ];
    }

    protected function handleApiResponse($response): array
    {
        if (!$response->ok()) {
            Log::error('Orangefer API Error', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return [
                'success' => false,
                'error' => 'HTTP ' . $response->status(),
                'message' => 'Request failed with HTTP error.',
            ];
        }

        $body = $response->json();

        // Check for known API codes
        if (isset($body['message'])) {
            return match ($body['message']) {
                '101' => [
                    'success' => false,
                    'error' => 'DUPLICATE_TRANSACTION',
                    'message' => 'Transaction is duplicated. Please use a unique reference.',
                ],
                '102' => [
                    'success' => false,
                    'error' => 'INSUFFICIENT_BALANCE',
                    'message' => 'You do not have enough balance.',
                ],
                '01020' => [
                    'success' => false,
                    'error' => 'CURRENCY_OR_PERMISSION_ERROR',
                    'message' => 'Currency mismatch or permission denied.',
                ],

                'bad request' => [
                    'success' => false,
                    'error' => 'BAD_REQUEST',
                    'message' => 'Invalid request sent to API.',
                ],
                default => [
                    'success' => false,
                    'error' => 'UNKNOWN_ERROR',
                    'message' => $body['message'],
                ],
            };
        }

        return [
            'success' => true,
            'ref' => $body['transnum'] ?? "",
            'password' => $body['password'] ?? "",
            'tax' => $body['tax'] ?? "",
            'tax_currency' => $body['taxcurrency'] ?? "",
            'message' => 'Transaction sent successfully.' ?? "",
        ];
    }


    public function testConnection(): bool
    {
        return $this->authenticate();
    }
}
