<?php
// app/Services/Partners/OrangeferApiService.php
namespace App\Services\Partners;

use Illuminate\Support\Facades\Log;
use FontLib\Table\Type\name;
use App\Interfaces\PartnerApiInterface;
use App\Models\Agent;
use App\Models\LedgerApproval;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;

class TurkuazApi implements PartnerApiInterface
{
    protected string $token;
    protected array $config;
    protected string $baseUrl;

    public function setConfig(array $config): void
    {
        $this->config = $config;
        $this->baseUrl = $config['base_url'];
    }



    public function authenticate(): bool
    {
        // Try to get token from cache
        $cachedToken = Cache::get('orangefer_api_token');

        if ($cachedToken) {
            $this->token = $cachedToken;
            return true;
        }
        // $this->token = "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJ3d3cua3JzNDduOTJ0LmNvbSIsImF1ZCI6Ind3dy5rcnM0N245MnQuY29tIiwiaWF0IjoxNzUzODY2MDMzLCJuYmYiOjE3NTM4NjYwMzMsImV4cCI6MTc1Mzk1MjQzMywiZGF0YSI6eyJJRCI6IjE1NTUiLCJIYXNoSUQiOiJjYmRmMmMyZGIxYjJlNTJjODE1MjA2OWJjZDQ5ZDIzNCIsIkxBIjotMTY3ODI5LjE1MjQ5OTk4ODMyNTg3MTUyNzE5NDk3NjgwNjY0MDYyNSwiTFQiOjE3NTM3Nzk2MzMsIkFBTSI6eyIxIjowLCIyIjowLCIzIjo4NTc3NzAuMTY5OTk5OTU3MDg0NjU1NzYxNzE4NzUsIjQiOi04OTE4MjUuMzQ5OTk5OTk0MDM5NTM1NTIyNDYwOTM3NX19fQ.pRhVODTN1yxqRggxLUWmDalI6QaS1HRlNaQP1nuk-vk";
        // return true;
        // Token not cached, perform authentication
        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'username' => $this->config['username'],
            'password' => $this->config['password'],
            'key' => $this->config['agentcode'],
        ])->post("{$this->baseUrl}/connect/index.php?p=l&f=i");

        Log::info('Orangefer Auth Response: ' . $response->body());

        if ($response->ok()) {
            $data = $response->json();

            if (isset($data['Token'])) {
                $this->token = $data['Token'];

                // Cache token for 55 minutes or appropriate expiry
                Cache::put('orangefer_api_token', $this->token, now()->addMinutes(55));

                return true;
            } else {
                Log::warning('Token missing from auth response', $data);
            }
        } else {
            Log::error('Orangefer auth failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
        }

        return false;
    }

    ///////////////////////////////Approval///////////////////////////////////
    public function getClientList(string $name, string $currency): array
    {
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/connect/index.php?p=c&f=namequery", [
            'name' => $name,
            'currency' => $currency,
        ]);

        $clients = $response->json() ?? [];
        Log::info('Turkuaz Client Response: ' . $response->body());
        return collect($clients)->map(function ($client) {
            return [
                'id' => $client['CID'],
                'text' => $client['CID'] . '-' . $client['CN'],
                'status' => 'active',
            ];
        })->all();
    }


    public function sendTransaction(array $data): array
    {

        $ref =  $data['reference'];
        if (!$this->authenticate()) {
            return []; // Or throw an exception
        }
        $data['sender_name'] = Agent::where('id', $data['sender_id'])->first()->name;
        $array = [];
        $array['ACID'] = $data['receiver_id'];
        $array['SenderName'] = $data['sender_name'];

        $array['ANote'] = $data['notes'];
        $array['ASAM'] = $data['amount'];
        $array['ASCur'] = $data['currency'];



        $result = Http::withHeaders([
            'Content-Type' => 'application/json',
            'X-Authorization' => 'Bearer ' . $this->token,
        ])->post("{$this->baseUrl}/connect/index.php?p=c&f=approval", $array);
        return $this->handleApiResponse($result);
    }
    public function getAllClients()
    {
        return [];
    }
    protected function handleApiResponse($response): array
    {
        if (!$response->ok()) {
            Log::error('Orangefer API Error', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return [
                'success' => false,
                'error' => 'HTTP ' . $response->status(),
                'message' => 'Request failed with HTTP error.',
            ];
        }

        $body = $response->json();

        // Check for known API codes
        if (isset($body['message'])) {
            return match ($body['message']) {
                '101' => [
                    'success' => false,
                    'error' => 'DUPLICATE_TRANSACTION',
                    'message' => 'Transaction is duplicated. Please use a unique reference.',
                ],
                '102' => [
                    'success' => false,
                    'error' => 'INSUFFICIENT_BALANCE',
                    'message' => 'You do not have enough balance.',
                ],
                '1020' => [
                    'success' => false,
                    'error' => 'CURRENCY_OR_PERMISSION_ERROR',
                    'message' => 'Currency mismatch or permission denied.',
                ],
                '2020' => [
                    'success' => false,
                    'error' => 'There is no account with requested ID',
                    'message' => 'There is no account with requested ID',
                ],
                'bad request' => [
                    'success' => false,
                    'error' => 'BAD_REQUEST',
                    'message' => 'Invalid request sent to API.',
                ],
                default => [
                    'success' => false,
                    'error' => 'UNKNOWN_ERROR',
                    'message' => $body['message'],
                ],
            };
        }

        return [
            'success' => true,
            'ref' => $body['Ref'],
        ];
    }


    public function testConnection(): bool
    {
        return $this->authenticate();
    }
}
