<?php

namespace App\Services\Sms;

use App\Services\Contracts\NotificationStrategy;
use App\Models\SmsSetting;
use App\Services\MessageTagParser;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Schema;

class SmsSender implements NotificationStrategy
{
  protected $config;

  public function __construct()
  {
    // Skip database operations in testing environment
    if (app()->environment('testing')) {
      return;
    }

    try {
      if (Schema::hasTable('sms_settings')) {
        $this->config = SmsSetting::first();
      }
    } catch (\Exception $e) {
      // Log error but don't fail
      Log::warning('SMS settings table not accessible', ['error' => $e->getMessage()]);
    }

    if (!$this->config) {
      //  throw new Exception("SMS configuration not found.");
    }
  }

  public function send(string $to, string $message, array $meta = []): bool
  {
    $parser = new MessageTagParser();
    $parsedMessage = $parser->parse($message, $meta);
    $partner = $meta['partner'] ?? 'System';

    $payloadRaw = str_replace(
      ['%TO%', '%MESSAGE%', '%PARTNER%'],
      [$to, $parsedMessage, $partner],
      $this->config->payload
    );

    if ($this->config->content_type === 'application/json') {
      $payload = json_decode($payloadRaw, true);
    } else {
      parse_str($payloadRaw, $payload);
    }

    $http = Http::withHeaders([
      'Content-Type' => $this->config->content_type,
    ]);

    $response = $this->config->method === 'GET'
      ? $http->get($this->config->gateway_url, $payload)
      : $http->post($this->config->gateway_url, $payload);

    $responseText = trim($response->body());

    if (
      $response->successful() &&
      $this->config->successfull_response &&
      str_contains($responseText, $this->config->successfull_response)
    ) {
      return true;
    }

    Log::error('SMS sending failed', [
      'url' => $this->config->gateway_url,
      'payload' => $payload,
      'response' => $responseText,
    ]);

    throw new Exception("SMS sending failed. Response: " . $responseText);
  }
  public  function getBalance()
  {
    $http = Http::withHeaders([
      'Content-Type' => $this->config->balance_response_type,
    ]);

    $response = $this->config->balance_method === 'GET'
      ? $http->get($this->config->balance_url)
      : $http->post($this->config->gateway_url);

    $responseText =  ($response->body());
    Log::info('SMS Balance', [
      'url' => $this->config->balance_url,
      'response' => $responseText,
      'response_all' => $response,
    ]);
  }
}
