<?php

namespace App\Services\Transfer;

use Illuminate\Support\Str;

use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Http\Controllers\JournalEntryController;
use Abivia\Ledger\Messages\Entry;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\JournalReferenceController;
use Abivia\Ledger\Messages\EntityRef;
use Abivia\Ledger\Messages\Reference;
use Abivia\Ledger\Models\JournalEntry;
use App\Models\Agent;
use App\Models\LedgerApproval;
use App\Models\LedgerBooking;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class ApprovalService
{

    public function create(array $data): LedgerApproval
    {


        $data['secret'] = rand(1000, 9999);
        $data['user_id'] = auth()->user()->id ?? 1;
        return LedgerApproval::create($data);
    }


    public function createTransferLedgerEntries(LedgerApproval $transfer): bool
    {

        try {

            $exchangeservice = new ExchangeService();
            $controller = new JournalEntryController();
            $timestamp = now()->format('Y-m-d H:i');

            // Create base reference
            $baseReference = $this->createBaseReference($transfer);



            if ($transfer->type == "approval") {

                $cash = $this->buildApprovalTransferDetails($transfer, $baseReference);
                $details = $cash;
                $groupedByCurrency = collect($details)->groupBy('currency');
                $groupedByCurrency = $groupedByCurrency->toArray();

                foreach ($groupedByCurrency as $currency => $details) {
                    $entry_array = Entry::fromArray([
                        'currency' => $currency, // 'TRY'
                        'entryType' => 'Transfer',
                        'clearing' => true,
                        'description' => "Transfer {$transfer->reference}",
                        'transDate' => $timestamp,
                        'clearing' => true,
                        'language' => App::getLocale(),
                        'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $transfer->id]),
                        'details' =>  $details,
                    ]);
                    $result = $controller->add($entry_array);
                    $result->transfer_id = $transfer->id;
                    $result->save();
                }


                // $status = "completed";
            }





            return true;
        } catch (Breaker $e) {
            //  return  false;
            dd([
                'exception_code' => $e->getCode(),
                'breaker_errors' => $e->getErrors(true), // Force all hidden errors
                'validation_rules' => config('ledger.rules'), // Check config
            ]);
        } catch (\Exception $e) {
            Log::error("Transfer failed: " . $e);
            return  false;
            // throw $e;
        }
    }



    public function createBaseReference(LedgerApproval $transfer): Reference
    {
        $reference = new Reference();
        $reference->code = $transfer->reference;
        $reference->domain = new EntityRef();
        $reference->domain->code = 'DEFAULT';
        $reference->extra = json_encode([
            'transfer_id' => $transfer->id,
            'type' => 'agent_transfer'
        ]);

        return $reference;
    }





    public function buildApprovalTransferDetails(LedgerApproval $transfer, Reference $baseReference): array
    {
        $accounts = $this->getValidatedAccounts($transfer);

        $rawDetails = [];

        $cashAmount = $transfer->amount + $transfer->send_fee;
        $company_fee = $transfer->send_fee;

        if ($cashAmount > 0) {
            $rawDetails[] = [
                'code' => $accounts['sender']['cash'],
                'currency' => $transfer->currency,
                'debit' => $cashAmount,
                'extra' => json_encode(['detail_type' => 'Approval_Initial', 'transfer_id' => $transfer->id]),
                'references' => [$this->createDetailReference($baseReference, 'Approval_Initial')],
            ];
        }


        $rawDetails[] = [
            'code' => $accounts['Undelivered_Approval'],
            'currency' => $transfer->delivery_currency,
            'credit' => $transfer->delivery_amount,
            'extra' => json_encode(['detail_type' => 'Approval_Initial', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'Approval_Initial')],
        ];


        if ($company_fee > 0) {
            $rawDetails[] = [
                'code' => $accounts['fees']['unearned_revenue'],
                'currency' => $transfer->currency,
                'credit' => $company_fee,
                'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
                'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')],
            ];
        }



        return array_values($rawDetails);
    }



    public  function UpdateLedger(LedgerApproval $transfer)
    {
        try {
            $scheme = request()->isSecure() ? 'https' : 'http';
            $host = request()->getHost();
            $entry = $transfer->journalEntries()->first();
            $entry_id = $entry->journalEntryId;

            $baseReference = $this->createBaseReference($transfer);
            $details = $this->buildApprovalTransferDetails($transfer,  $baseReference);
            $extra = json_decode($entry->extra, true) ?? [];
            $extra['is_updated'] = true;
            $response = Http::post("{$scheme}://{$host}/api/ledger/entry/get", [
                'id' => $entry_id,
            ]);
            $ledgerEntry = $response->json();

            $ledgerEntry = $ledgerEntry['entry'];


            $revision = $ledgerEntry['revision'];

            $response = Http::post("{$scheme}://{$host}/api/ledger/entry/update", [
                'id' => $entry_id,
                'revision' => $revision,
                'currency' => $transfer->currency, // or dynamic
                'date' => $transfer->updated_at->toDateString(), // or use now
                'description' => 'Updated Approval #' . $transfer->id,
                'extra' => $extra,

                'details' =>  $details
            ]);

            return (['success' => true, 'message' => __("alerts.saved_successfully")]);
        } catch (\Exception $e) {
            Log::info($e);
            return  ['success' => false, 'message' => __("alerts.something_went_wrong")];
        }
    }




    public function debugBalance(string $title, array $details)
    {
        $sum = 0;
        $text = "$title Details:\n";

        foreach ($details as $d) {
            $amount = $d['debit'] ?? ($d['credit'] ?? 0);
            $type = isset($d['debit']) ? 'Debit' : 'Credit';
            $sum += isset($d['debit']) ? $amount : -$amount;

            // Fetch the account name
            $account = LedgerAccount::with("names")->where('code', $d['code'])->first();

            $accountName = $account ? $account->names->first()->name : 'Unknown Account';

            $text .= "- {$d['code']} ({$accountName}): {$type} {$amount}\n";
        }

        $text .= "Total Balance: $sum\n";
        $text .= "------------------\n";

        return $text;
    }


    public function createDetailReference(Reference $baseReference, string $suffix): Reference
    {
        $reference = new Reference();
        $reference->code = $baseReference->code . '-' . $suffix;
        $reference->domain = clone $baseReference->domain;
        $reference->extra = json_encode(array_merge(
            json_decode($baseReference->extra, true) ?? [],
            ['detail_type' => $suffix]
        ));

        //     $controller = new JournalReferenceController();
        //    $reference = $controller->add($reference);


        return $reference;
    }


    public function getValidatedAccounts(LedgerApproval $transfer, $delivery_agent = null): array
    {

        $send_currency = $transfer->currency ?? "USD";
        $delivery_currency = $transfer->delivery_currency ?? $send_currency;
        $data = [
            'sender' => [
                'cash' => $this->getAgentAccountCode($transfer->sender_id, '11', $send_currency), // حساب الوكيل

                'commission' => 5100, //$this->getAgentAccountCode($transfer->sender_id, '51') // نضع فيها  الارباح  للعميل  كمصروفات 
            ],
            'receiver' => [
                'cash' => $transfer->receiver_account_code ?? $this->getAgentAccountCode($transfer->receiver_id, '11', $delivery_currency), // حساب الوكيل

                'commission' => "5100" //$this->getAgentAccountCode($transfer->receiver_id, '51')

            ],
            'delivery_agent' => [

                'cash' => $transfer->receiver_account_code ?? getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '11', $delivery_currency), // حساب الوكيل

                'unearned_revenue' => "5200",
                'commission' => "5100" // $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '51')

            ],

            'fees' => [
                'unearned_revenue' => "5200",
                'commission' => "5100",


                // 'revenue' => $this->getAgentAccountCode($transfer->receiver_id, '41'),
                // 'expense' => $this->getAgentAccountCode($transfer->sender_id, '51')

            ],
            'Undelivered_exchange' => "1410",
            'Undelivered_Approval' => "1300"

        ];

        return $data;
    }







    public function getAgentAccountCode(int $agentId, string $prefix): string
    {
        $agent = Agent::findOrFail($agentId);
        $real_agent_id = $agent->rootAgent()->id;
        $account = LedgerAccount::where('extra->agent_id', $real_agent_id)
            ->where('code', 'like', $prefix . '%')
            ->firstOrFail();

        return $account->code;
    }

    public function generateRandomTransferNumber(): string
    {
        $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
        $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

        return "APR-{$branchCode}-{$transactionId}";
    }
}
