<?php

namespace App\Services\Transfer;

use Illuminate\Support\Str;

use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Http\Controllers\JournalEntryController;
use Abivia\Ledger\Messages\Entry;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\JournalReferenceController;
use Abivia\Ledger\Messages\EntityRef;
use Abivia\Ledger\Messages\Reference;
use Abivia\Ledger\Models\JournalEntry;
use App\Models\Agent;
use App\Models\LedgerBooking;
use App\Models\LedgerCashier;
use App\Models\LedgerExchange;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class CashierLedgerService
{




    public function createCashierLedgerEntries(LedgerCashier $transfer, $type = "pay"): bool
    {

        try {


            $exchangeservice = new ExchangeService();
            $controller = new JournalEntryController();
            $timestamp = now()->format('Y-m-d H:i');

            // Create base reference
            $baseReference = $this->createBaseReference($transfer);

            Log::info("Creating transfer ledger entries for transfer ID: {$transfer->id}, reference: {$transfer->reference}");


            if ($type != "pay")
                $cash = $this->buildCashierTransferDetails($transfer, $baseReference, $type);
            else
                $cash = $this->buildCashierBookingTransferDetails($transfer, $baseReference);

            $details = $cash;

            $groupedByCurrency = collect($details)->groupBy('currency');
            $groupedByCurrency = $groupedByCurrency->toArray();

            foreach ($groupedByCurrency as $currency => $details) {
                $entry_array = Entry::fromArray([
                    'currency' => $currency, // 'TRY'
                    'entryType' => 'Transfer',
                    'clearing' => true,
                    'description' => "Transfer {$transfer->reference}",
                    'transDate' => $timestamp,
                    'clearing' => true,
                    'language' => App::getLocale(),
                    'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $transfer->id]),
                    'details' =>  $details,
                ]);

                if ($transfer->status == "reverse") {
                    $entry_array->extra = json_encode([
                        'type' => 'reversal',
                        'reverses_transfer_id' => $transfer->extra['reverses_transfer_id'] ?? null
                    ]);
                }
                $result = $controller->add($entry_array);
                $result->transfer_id = $transfer->id;
                $result->save();
            }


            if ($transfer->status != "reverse") {
                $status = "completed";
                $transfer->status = $status;
            }
            $transfer->save();



            return true;
        } catch (Breaker $e) {
            //  return  false;
            dd([
                'exception_code' => $e->getCode(),
                'breaker_errors' => $e->getErrors(true), // Force all hidden errors
                'validation_rules' => config('ledger.rules'), // Check config
            ]);
        } catch (\Exception $e) {
            Log::error("Transfer failed: " . $e);

            throw $e;
            return  false;
        }
    }



    public function createBaseReference(LedgerCashier $transfer): Reference
    {
        $reference = new Reference();
        $reference->code = $transfer->reference;
        $reference->domain = new EntityRef();
        $reference->domain->code = 'DEFAULT';
        $reference->extra = json_encode([
            'transfer_id' => $transfer->id,
            'type' => 'agent_transfer'
        ]);

        return $reference;
    }





    public function buildCashierTransferDetails(LedgerCashier $transfer, Reference $baseReference, $type): array
    {

        $accounts = $this->getValidatedAccounts($transfer);

        $rawDetails = [];

        $cashAmount = $transfer->delivery_amount; // + $transfer->send_fee;
        $currency = $transfer->delivery_currency;

        if ($type == "collect") {
            $cashAmount = $transfer->amount + $transfer->send_fee;
            $currency = $transfer->currency;
        }
        if ($cashAmount > 0) {
            $rawDetails[] = [
                'code' => $accounts['sender']['cash'],
                'currency' => $currency,
                'debit' => $cashAmount,
                'extra' => json_encode(['detail_type' => 'INITIAL_TRANSFER', 'transfer_id' => $transfer->id]),
                'references' => [$this->createDetailReference($baseReference, 'INITIAL_TRANSFER')],
            ];
        }


        $rawDetails[] = [
            'code' => $accounts['receiver']['cash'],
            'currency' => $currency,
            'credit' => $cashAmount,
            'extra' => json_encode(['detail_type' => 'DELIVERY_TRANSFER', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
        ];





        return array_values($rawDetails);
    }

    public function buildCashierBookingTransferDetails(LedgerCashier $transfer, Reference $baseReference): array
    {
        $send_currency = $transfer->currency ?? "USD";
        $delivery_currency = $transfer->delivery_currency ?? $send_currency;
        $data = [
            'sender' => [
                'cash' => $this->getValidatedAccounts($transfer->sender_id, '11'), // حساب الوكيل
                'not_delivered' => "1300", ////$this->getAgentAccountCode($transfer->sender_id, '13'),
                'commission' => "5100", //$this->getAgentAccountCode($transfer->sender_id, '51') // نضع فيها  الارباح  للعميل  كمصروفات 
                'unearned_revenue' => "5200",
            ],
            'receiver' => [
                'cash' => $this->getValidatedAccounts($transfer->receiver_id, '11'), // حساب الوكيل
                'not_delivered' => "1300", //$this->getAgentAccountCode($transfer->receiver_id, '13'),
                'commission' => "5100", //$this->getAgentAccountCode($transfer->receiver_id, '51')
                'unearned_revenue' => "5200",

            ],
            'delivery_agent' => [

                'cash' =>  $this->getValidatedAccounts($delivery_agent ?? $transfer->receiver_id, '11'), // حساب الوكيل
                'not_delivered' => "1300",
                'unearned_revenue' => "5200",
                'commission' => "5100" // $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '51')

            ],

            'fees' => [
                'unearned_revenue' => "5200",
                'commission' => "5100",


                // 'revenue' => $this->getAgentAccountCode($transfer->receiver_id, '41'),
                // 'expense' => $this->getAgentAccountCode($transfer->sender_id, '51')

            ],
            'Undelivered_exchange' => "1410"

        ];

        return $data;
    }



    public function debugBalance(string $title, array $details)
    {
        $sum = 0;
        $text = "$title Details:\n";

        foreach ($details as $d) {
            $amount = $d['debit'] ?? ($d['credit'] ?? 0);
            $type = isset($d['debit']) ? 'Debit' : 'Credit';
            $sum += isset($d['debit']) ? $amount : -$amount;

            // Fetch the account name
            $account = LedgerAccount::with("names")->where('code', $d['code'])->first();

            $accountName = $account ? $account->names->first()->name : 'Unknown Account';

            $text .= "- {$d['code']} ({$accountName}): {$type} {$amount}\n";
        }

        $text .= "Total Balance: $sum\n";
        $text .= "------------------\n";

        return $text;
    }


    public function createDetailReference(Reference $baseReference, string $suffix): Reference
    {
        $reference = new Reference();
        $reference->code = $baseReference->code . '-' . $suffix;
        $reference->domain = clone $baseReference->domain;
        $reference->extra = json_encode(array_merge(
            json_decode($baseReference->extra, true) ?? [],
            ['detail_type' => $suffix]
        ));



        return $reference;
    }


    public function getValidatedAccounts(LedgerCashier $transfer, $delivery_agent = null): array
    {



        $send_currency = $transfer->currency ?? "USD";
        $delivery_currency = $transfer->delivery_currency ?? $send_currency;
        $data = [
            'sender' => [
                'cash' => $this->getCashierAccountCode($transfer->sender_id, '11', $send_currency), // حساب الوكيل


            ],
            'receiver' => [
                'cash' => $this->getCashierAccountCode($transfer->receiver_id, '11', $delivery_currency), // حساب الوكيل
            ],


        ];

        return $data;
    }







    public function getCashierAccountCode(int $agentId, string $prefix): string
    {
        $agent = Agent::findOrFail($agentId);
        $real_agent_id = $agent->id;
        $account = LedgerAccount::where('extra->agent_id', $real_agent_id)
            ->where('code', 'like', $prefix . '%')
            ->firstOrFail();

        return $account->code;
    }

    public function generateRandomTransferNumber(): string
    {
        $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
        $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

        return "REC-{$branchCode}-{$transactionId}";
    }
}
