<?php

namespace App\Services\Transfer;

use Illuminate\Support\Str;
use App\Models\LedgerTransfer;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Http\Controllers\JournalEntryController;
use Abivia\Ledger\Messages\Entry;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\JournalReferenceController;
use Abivia\Ledger\Messages\EntityRef;
use Abivia\Ledger\Messages\Reference;
use Abivia\Ledger\Models\JournalEntry;
use App\Models\Agent;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class TransferService
{
  public function create(array $data): LedgerTransfer
  {




    return LedgerTransfer::create($data);
  }
  public  function UpdateLedger($transfer)
  {
    try {
      $scheme = request()->isSecure() ? 'https' : 'http';
      $host = request()->getHost();
      $entry_id = $transfer->journalEntries()->first()->journalEntryId;
      $baseReference = $this->createBaseReference($transfer);
      $details = $this->buildInitialTransferDetails($transfer,  $baseReference);
      $response = Http::post("{$scheme}://{$host}/api/ledger/entry/get", [
        'id' => $transfer->journalEntries()->first()->journalEntryId,
      ]);
      $ledgerEntry = $response->json();

      $ledgerEntry = $ledgerEntry['entry'];


      $revision = $ledgerEntry['revision'];

      $response = Http::post("{$scheme}://{$host}/api/ledger/entry/update", [
        'id' => $entry_id,
        'revision' => $revision,
        'currency' => $transfer->currency, // or dynamic
        'date' => $transfer->updated_at->toDateString(), // or use now
        'description' => 'Updated Transfer #' . $transfer->id,
        'details' =>  $details
      ]);

      return (['success' => true, 'message' => 'Transfer and ledger updated.']);
    } catch (\Exception $e) {
      return  ['success' => false, 'message' => 'Error updating transfer: ' . $e->getMessage()];
    }
  }
  public  function DeleteLedger($transfer)
  {
    try {
      $scheme = request()->isSecure() ? 'https' : 'http';
      $host = request()->getHost();
      $entries = $transfer->journalEntries;
      foreach ($entries as $entry) {
        $entry_id = $entry->journalEntryId;

        $response = Http::post("{$scheme}://{$host}/api/ledger/entry/get", [
          'id' => $entry_id,
        ]);
        $ledgerEntry = $response->json();

        $ledgerEntry = $ledgerEntry['entry'];


        $revision = $ledgerEntry['revision'];

        $response = Http::post("{$scheme}://{$host}/api/ledger/entry/delete", [
          'id' => $entry_id,
          'revision' => $revision,

        ]);
      }
      return (['success' => true, 'message' => __("alerts.saved_successfully")]);
    } catch (\Exception $e) {
      Log::info($e);
      return  ['success' => false, 'message' => __("alerts.something_went_wrong")];
    }
  }

  public function reverseTransferLedgerEntries($transfer): bool
  {
    try {
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');
      $baseReference = $this->createBaseReference($transfer);

      // Get original details (e.g. re-run the same logic)
      $originalDetails = $this->buildInitialTransferDetails($transfer, $baseReference);

      // Reverse the details
      $reversedDetails = collect($originalDetails)->map(function ($detail) {
        $newDetail = $detail;

        if (isset($detail['debit'])) {
          $newDetail['credit'] = $detail['debit'];
          unset($newDetail['debit']);
        } elseif (isset($detail['credit'])) {
          $newDetail['debit'] = $detail['credit'];
          unset($newDetail['credit']);
        }

        return $newDetail;
      })->toArray();

      $groupedByCurrency = collect($reversedDetails)->groupBy('currency')->toArray();

      foreach ($groupedByCurrency as $currency => $details) {
        $entry_array = Entry::fromArray([
          'currency' => $currency,
          'entryType' => 'Transfer Reversal',
          'description' => "Reversal of transfer {$transfer->reference}",
          'transDate' => $timestamp,
          'language' => App::getLocale(),
          'extra' => json_encode([
            'type' => 'reversal',
            'reverses_transfer_id' => $transfer->id
          ]),
          'details' => $details,

        ]);

        $result = $controller->add($entry_array);
        $result->transfer_id = $transfer->id;
        $result->save();
      }

      return true;
    } catch (Breaker $e) {
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true),
        'validation_rules' => config('ledger.rules'),
      ]);
    } catch (\Exception $e) {
      Log::error("Reverse transfer failed: " . $e);
      return false;
    }
  }

  public function createTransferLedgerEntries($transfer): bool
  {

    try {


      $exchangeservice = new ExchangeService();
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($transfer);


      if ($transfer->type == "transfer") {
        $cash = $this->buildInitialTransferDetails($transfer, $baseReference);


        // if ($transfer->currency != $transfer->delivery_currency) {
        //     $details = $exchangeservice->BuildExchangeLedgerDetails($transfer->currency, $transfer->delivery_currency, $transfer->amount, $transfer->delivery_amount, $transfer);
        //     $details = array_merge(
        //         $cash,
        //         $details
        //     );
        // } else {
        //     $details = $cash;
        // }


        $details = $cash;



        $groupedByCurrency = collect($details)->groupBy('currency');
        $groupedByCurrency = $groupedByCurrency->toArray();
        $description = generateTransferDescription($transfer);
        Log::info("description: " . $description);
        foreach ($groupedByCurrency as $currency => $details) {
          $entry_array = Entry::fromArray([
            'currency' => $currency, // 'TRY'
            'entryType' => 'Transfer',
            'description' => $description, // "Transfer {$transfer->reference}",
            'transDate' => $timestamp,
            'clearing' => true,
            'language' => App::getLocale(),
            'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $transfer->id]),
            'details' =>  $details,
            'transfer_id' => $transfer->id
          ]);
          $res = $controller->add($entry_array);
          $res->transfer_id = $transfer->id;
          $res->save();
        }
        return true;

        $status = "pending";
      }
      if ($transfer->type == "booking") {
        $bookingservice = new BookingService();
        $bookingservice->createTransferLedgerEntries($transfer);
      }
      if ($transfer->type == "approval") {
        $approvalservice = new ApprovalService();
        $approvalservice->createTransferLedgerEntries($transfer);
      }

      // $sendResponse = $controller->add($senderEntry);
      // $transfer->status = $status;
      // $transfer->save();

      // $transfer->update([
      //     'debit_journal_uuid' => $sendResponse->journalEntryId,
      //     'status' => $status,
      //     'processed_at' => $timestamp
      // ]);


      return true;
    } catch (Breaker $e) {
      //  return  false;
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      return  false;
      // throw $e;
    }
  }
  public function updateTransferLedgerEntries($transfer): bool
  {
    try {
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i:s');

      // Validate required properties
      if (!$transfer->debit_journal_uuid) {
        throw new \Exception("Missing journal UUID for update");
      }

      // Retrieve existing entry to get revision hash
      $existingEntry = $controller->get($transfer->debit_journal_uuid)->journalEntry;

      // Rebuild transaction details
      $baseReference = $this->createBaseReference($transfer);

      if ($transfer->type == "transfer") {
        $details = $this->buildInitialTransferDetails($transfer, $baseReference);
        $description = "Transfer {$transfer->reference} (Sender)";
      } elseif ($transfer->type == "booking") {
        $details = $this->buildBookingTransferDetails($transfer, $baseReference);
        $description = "Agent {$transfer->reference}";
      } else {
        throw new \Exception("Invalid transfer type: {$transfer->type}");
      }


      // Build updated entry
      $updatedEntry = Entry::fromArray([
        'id' => $transfer->debit_journal_uuid,
        'revision' => $existingEntry->revision, // Critical for optimistic locking
        'currency' => $transfer->currency,
        'clearing' => true,
        'entryType' => 'transfer',
        'description' => $description,
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'details' => $details,
        'extra' => ($transfer->type == "booking")
          ? json_encode(['type' => 'agent_transfer', 'transfer_id' => $transfer->id])
          : null,
        'transfer_id' => $transfer->id
      ]);

      // Update the entry
      $result = $controller->update($updatedEntry);
      $result->transfer_id = $transfer->id;
      $result->save();

      // Update transfer timestamps
      $transfer->update([
        'processed_at' => $timestamp,
        'updated_at' => $timestamp
      ]);

      return true;
    } catch (Breaker $e) {
      Log::error("Ledger update failed: " . $e->getMessage(), [
        'errors' => $e->getErrors(true),
        'transfer_id' => $transfer->id
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error("Transfer update failed: " . $e->getMessage(), [
        'transfer_id' => $transfer->id
      ]);
      return false;
    }
  }
  public  function  DeliverTransfer($transfer)
  {

    try {
      if ($transfer->status !== 'pending') {
        throw new \Exception("Transfer must be in pending status");
      }
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');
      $exchange_service = new ExchangeService();

      // Create base reference
      $baseReference = $this->createBaseReference($transfer);
      $delivery =  $this->buildDeliveryDetails($transfer, $baseReference);
      $exchange = [];
      if ($transfer->currency != $transfer->delivery_currency)
        $exchange = $exchange_service->BuildExchangeLedgerDetails($transfer->currency, $transfer->delivery_currency, $transfer->amount, $transfer->delivery_amount, $transfer);
      $revenue = $this->BuildTransferRevenueDistributionDetails($transfer, $baseReference);

      $details = array_merge(
        $delivery,
        $exchange,
        $revenue
      );

      $groupedByCurrency = collect($details)->groupBy('currency');
      $groupedByCurrency = $groupedByCurrency->toArray();

      foreach ($groupedByCurrency as $currency => $details) {
        $entry_array = Entry::fromArray([
          'currency' => $currency, // 'TRY'
          'entryType' => 'Transfer',
          'description' => "Transfer {$transfer->reference}",
          'transDate' => $timestamp,
          'clearing' => true,
          'language' => App::getLocale(),
          'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $transfer->id]),
          'details' =>  $details,
        ]);
        $res = $controller->add($entry_array);

        $res->transfer_id = $transfer->id;
        $res->save();
      }

      $transfer->update([

        //  'credit_journal_uuid' => $receiveResponse->journalEntryId,
        'status' => 'completed',
        'delivered_at' => $timestamp
      ]);
      return true;
    } catch (Breaker $e) {
      Log::error("Ledger update failed: " . $e->getMessage(), [
        'errors' => $e->getErrors(true),
        'transfer_id' => $transfer->id
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e->getMessage());
      throw $e;
      return false;
    }
  }

  public function createBaseReference($transfer): Reference
  {
    $reference = new Reference();
    $reference->code = $transfer->reference;
    $reference->domain = new EntityRef();
    $reference->domain->code = 'DEFAULT';
    $reference->extra = json_encode([
      'transfer_id' => $transfer->id,
      'type' => 'agent_transfer'
    ]);

    return $reference;
  }

  public function buildSenderDetails($transfer, Reference $baseReference): array
  {
    $accounts = $this->getValidatedAccounts($transfer);

    return [
      // Customer deposits cash (Istanbul agent receives money)
      [
        'code' => $accounts['sender']['cash'], // 1100
        'debit' => $transfer->amount + $transfer->sender_fee,
        'references' => [$this->createDetailReference($baseReference, 'CUSTOMER_DEPOSIT')]
      ],
      // Undelivered transfer liability
      [
        'code' => $accounts['receiver']['not_delivered'], // 1300
        'credit' => $transfer->amount,
        'references' => [$this->createDetailReference($baseReference, 'UNDELIVERED_TRANSFER')]
      ],
      // Unearned fee revenue
      [
        'code' => $accounts['sender']['unearned_revenue'], // 2200
        'credit' => $transfer->sender_fee,
        'references' => [$this->createDetailReference($baseReference, 'UNEARNED_FEE')]
      ]
    ];
  }
  public function buildInitialTransferDetails($transfer, Reference $baseReference): array
  {

    $accounts = $this->getValidatedAccounts($transfer);

    $details = [];

    $cashAmount = $transfer->amount + $transfer->send_fee;
    if ($cashAmount > 0) {
      $details[] = [
        'code' => $accounts['sender']['cash'],
        'currency' => $transfer->currency,
        'debit' => $cashAmount,
        'references' => [$this->createDetailReference($baseReference, 'INITIAL_TRANSFER')]
      ];
    }

    if ($transfer->amount > 0) {
      $details[] = [
        'code' => $accounts['receiver']['not_delivered'],
        'currency' => $transfer->currency,
        'credit' => $transfer->amount,
        'references' => [$this->createDetailReference($baseReference, 'INITIAL_TRANSFER')]
      ];
    }

    if ($transfer->send_fee != 0) {
      $details[] = [
        'code' => $accounts['fees']['unearned_revenue'],
        'currency' => $transfer->currency,
        'credit' => $transfer->send_fee,
        'references' => [$this->createDetailReference($baseReference, 'UNEARNED_FEE')]
      ];
    }
    // if ($transfer->currency != $transfer->delivery_currency) {
    //     $details[] = [
    //         'code' => $accounts['Undelivered_exchange'],
    //         'currency' => $transfer->currency,
    //         'credit' => $transfer->amount,
    //     ];
    //     $details[] = [

    //         'code' => $accounts['Undelivered_exchange'],
    //         'currency' => $transfer->delivery_currency,
    //         'debit' => $transfer->delivery_amount,
    //     ];
    // }


    return $details;
  }

  public  function buildTransferDeliveryDetails($transfer, Reference $baseReference): array
  {
    $accounts = $this->getValidatedAccounts($transfer);
    $company_fee = $transfer->send_fee - $transfer->receiver_fee;

    return [
      // Wrap your entries in an 'entries' array
      [
        'code' => $accounts['receiver']['not_delivered'],
        'debit' => $transfer->delivery_amount,
        'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
      ],
      [
        'code' => $accounts['receiver']['cash'],
        'credit' => $transfer->delivery_amount,
        'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
      ],
      [
        'code' => $accounts['fees']['unearned_revenue'],
        'debit' => $transfer->send_fee,
        'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
      ],
      [
        'code' => $accounts['receiver']['commission'],
        'credit' => $transfer->receiver_fee,
        'references' => [$this->createDetailReference($baseReference, 'ReCEIVER_COMMISSION')]
      ],
      [
        'code' => $accounts['fees']['commission'],
        'credit' => $company_fee,
        'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
      ]

    ];
  }


  public function buildBookingTransferDetails($transfer, Reference $baseReference): array
  {
    $accounts = $this->getValidatedAccounts($transfer);

    $details = [];

    $cashAmount = $transfer->amount + $transfer->send_fee;
    $company_fee = $transfer->send_fee;

    if ($cashAmount > 0) {
      $details[] = [
        'code' => $accounts['sender']['cash'],
        'currency' => $transfer->currency,
        'debit' => $cashAmount,
        'extra' => json_encode(['detail_type' => 'INITIAL_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'INITIAL_TRANSFER')]
      ];
    }
    if ($transfer->delivery_amount > 0) {
      $details[] = [
        'code' => $accounts['receiver']['cash'],
        'currency' => $transfer->delivery_currency,
        'credit' => $transfer->delivery_amount + $transfer->receiver_fee,
        'extra' => json_encode(['detail_type' => 'DELIVERY_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
      ];
    }

    if ($company_fee > 0) {


      $details[] =   [
        'code' => $accounts['fees']['commission'],
        'currency' => $transfer->currency,
        'credit' => $transfer->send_fee,
        'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
      ];
      if ($transfer->receiver_fee > 0) {
        $details[] =   [
          'code' => $accounts['receiver']['commission'],
          'currency' => $transfer->delivery_currency,
          'debit' => $transfer->receiver_fee,
          'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
          'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
        ];
      }
    }



    return $details;
  }




  public function buildReceiverDetails($transfer, Reference $baseReference): array
  {
    $accounts = $this->getValidatedAccounts($transfer);
    $company_fee = $transfer->sender_fee - $transfer->receiver_fee;
    return [
      // Amount owed to Aleppo agent (minus their commission)
      [
        'code' => $accounts['receiver']['payable'], // 2100
        'credit' => $transfer->amount + $transfer->receiver_fee,
        'references' => [$this->createDetailReference($baseReference, 'AGENT_PAYABLE')]
      ],

      // Commission expense
      [
        'code' => $accounts['receiver']['commission'], // 5100
        'debit' => $transfer->receiver_fee,
        'references' => [$this->createDetailReference($baseReference, 'AGENT_COMMISSION')]
      ],
      // Corresponding undelivered transfer
      [
        'code' => $accounts['fees']['not_delivered'], // 1300
        'debit' => $transfer->amount,
        'references' => [$this->createDetailReference($baseReference, 'UNDELIVERED_TRANSFER')]
      ]
    ];
  }

  public function buildDeliveryDetails($transfer, Reference $baseReference, $delivery_agent = null,): array
  {

    $accounts = $this->getValidatedAccounts($transfer, $delivery_agent);

    $details = [];

    $details[] = [
      'code' => $accounts['receiver']['not_delivered'], // 1300
      'debit' => $transfer->amount,
      'currency' => $transfer->currency,
      'references' => [$this->createDetailReference($baseReference, 'CLEAR_UNDELIVERED')]
    ];

    $details[] =
      [
        'code' => $accounts['delivery_agent']['cash'], // 1300
        'credit' => $transfer->delivery_amount,
        'currency' => $transfer->delivery_currency,
        'references' => [$this->createDetailReference($baseReference, 'CLEAR_UNDELIVERED')]
      ];

    // if ($transfer->currency != $transfer->delivery_currency) {
    //     $details[] = [
    //         'code' => $accounts['Undelivered_exchange'],
    //         'currency' => $transfer->currency,
    //         'debit' => $transfer->amount,
    //     ];
    //     $details[] = [

    //         'code' => $accounts['Undelivered_exchange'],
    //         'currency' => $transfer->delivery_currency,
    //         'credit' => $transfer->delivery_amount,
    //     ];
    // }

    return $details;
  }
  public function BuildTransferRevenueDistributionDetails($transfer, Reference $baseReference, $delivery_agent = null): array
  {
    $fee = $transfer->send_fee;
    $sender = $transfer->sender;
    $receiver = $transfer->receiver->rootAgent();
    if ($delivery_agent) {
      $receiver = Agent::find($delivery_agent);
    }

    if ($transfer->type == "transfer") {
      $sender_revenue_share = $sender->revenuepercent()->first()->outcome ?? 0;
      $receiver_revenue_share = $receiver?->revenuepercent()->first()->income ?? 0;
      $company_percent = 100 - $sender_revenue_share - $receiver_revenue_share;
      $company_fee = $company_percent * $fee / 100;
      $sender_fee = $sender_revenue_share * $fee / 100;
      $receiver_fee = $receiver_revenue_share * $fee / 100;
    } else {
      $sender_fee = 0;
      $receiver_fee = 0;
      $company_fee = $fee;
    }

    $accounts = $this->getValidatedAccounts($transfer);
    $details = [];
    if ($fee != 0) {
      $details[] = [
        'code' => $accounts['sender']['unearned_revenue'],
        'currency' => $transfer->currency,
        'debit' => $transfer->send_fee,
        'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
      ];
      if ($sender_fee != 0) {
        $details[] =
          [
            'code' => $accounts['sender']['commission'],
            'currency' => $transfer->currency,
            'credit' => $sender_fee,
            'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
          ];
      }
      if ($receiver_fee != 0) {
        $details[] =
          [
            'code' => $accounts['delivery_agent']['commission'],
            'currency' => $transfer->currency,
            'credit' => $receiver_fee,
            'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
          ];
      }
      if ($company_fee != 0) {
        $details[] =
          [
            'code' => $accounts['fees']['commission'],
            'currency' => $transfer->currency,
            'credit' => $company_fee,
            'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')]
          ];
      }
    }

    return $details;
  }
  public function debugBalance(string $title, array $details)
  {
    $sum = 0;
    $text = "$title Details:\n";

    foreach ($details as $d) {
      $amount = $d['debit'] ?? ($d['credit'] ?? 0);
      $type = isset($d['debit']) ? 'Debit' : 'Credit';
      $sum += isset($d['debit']) ? $amount : -$amount;

      // Fetch the account name
      $account = LedgerAccount::with("names")->where('code', $d['code'])->first();

      $accountName = $account ? $account->names->first()->name : 'Unknown Account';

      $text .= "- {$d['code']} ({$accountName}): {$type} {$amount}\n";
    }

    $text .= "Total Balance: $sum\n";
    $text .= "------------------\n";

    return $text;
  }


  public function createDetailReference(Reference $baseReference, string $suffix): Reference
  {
    $reference = new Reference();
    $reference->code = $baseReference->code . '-' . $suffix;
    $reference->domain = clone $baseReference->domain;
    $reference->extra = json_encode(array_merge(
      json_decode($baseReference->extra, true) ?? [],
      ['detail_type' => $suffix]
    ));

    //     $controller = new JournalReferenceController();
    //    $reference = $controller->add($reference);


    return $reference;
  }


  public function getValidatedAccounts($transfer, $delivery_agent = null): array
  {
    $send_currency = $transfer->currency ?? "USD";
    $delivery_currency = $transfer->delivery_currency ?? $send_currency;
    $data = [
      'sender' => [
        'cash' => getAgentAccountCode($transfer->sender_id, '11'), // حساب الوكيل


        'not_delivered' => "1300", //$this->getAgentAccountCode($transfer->sender_id, '13'),
        // 'revenue' => $this->getAgentAccountCode($transfer->sender_id, '41'), //الارباح للشركة
        'unearned_revenue' => "1310",
        'commission' => $this->getAgentAccountCode($transfer->sender_id, '1200') // نضع فيها  الارباح  للعميل  كمصروفات
      ],
      'receiver' => [
        'cash' => getAgentAccountCode($transfer->receiver_id, '11'), // حساب الوكيل

        //  'payable' => $this->getAgentAccountCode($transfer->receiver_id, '21' ),
        'not_delivered' =>  "1300", //this->getAgentAccountCode($transfer->receiver_id, '13'),
        //  'revenue' => $this->getAgentAccountCode($transfer->receiver_id, '41'),
        // 'unearned_revenue' => "5200",
        'commission' => $this->getAgentAccountCode($transfer->receiver_id, '51')

      ],
      'delivery_agent' => [

        'cash' => getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '11', $delivery_currency), // حساب الوكيل
        // 'receivable' => $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '12', $delivery_currency), //المستحقات من العميل
        // 'payable' => $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '21', $delivery_currency),
        'not_delivered' => "1300", // $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '13'),
        // 'revenue' => $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '41'),
        'unearned_revenue' => "1310",
        'commission' => $this->getAgentAccountCode($delivery_agent ?? $transfer->receiver_id, '1200')

      ],

      'fees' => [
        'unearned_revenue' => "1310",
        'commission' => "4110",


        // 'revenue' => $this->getAgentAccountCode($transfer->receiver_id, '41'),
        //  'expense' => $this->getAgentAccountCode($transfer->sender_id, '51')

      ],
      'Undelivered_exchange' => "1410"

    ];

    return $data;
  }







  public function getAgentAccountCode($agentId, string $prefix): string
  {
    $agent = Agent::findOrFail($agentId);
    $real_agent_id = $agent->rootAgent()->id;
    $account = LedgerAccount::where('extra->agent_id', $real_agent_id)
      ->where('code', 'like', $prefix . '%')
      ->first();

    return $account->code ?? "";
  }

  public function generateRandomTransferNumber(): string
  {
    $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "TRF-{$branchCode}-{$transactionId}";
  }
}
