/**
 *  Logistics Dashboard
 */

(function () {
  let borderColor, labelColor, headingColor, legendColor;
  let BudgetChart;
  let CategoryChart;
  let CashFlowChart;
  let propertyTypeChart;
  const isDarkStyle = $('html').hasClass('dark-style') || $('html').hasClass('semi-dark-style');
  if (isDarkStyle) {
    borderColor = config.colors_dark.borderColor;
    labelColor = config.colors_dark.textMuted;
    headingColor = config.colors_dark.headingColor;
    legendColor = config.colors_dark.bodyColor;
  } else {
    borderColor = config.colors.borderColor;
    labelColor = config.colors.textMuted;
    headingColor = config.colors.headingColor;
    legendColor = config.colors.bodyColor;
  }

  // Chart Colors
  const chartColors = {
    donut: {
      series1: config.colors.success,
      series2: config.colors.primary,
      series3: config.colors.warning,
      series4: config.colors.danger,
      series5: '#7EDDA9',
      series6: '#A9E9C5'
    },
    line: {
      series1: config.colors.warning,
      series2: config.colors.primary,
      series3: '#7367f029'
    }
  };

  window.addEventListener('InvoiceChartDataUpdated', function (event) {
    var data = event.detail[0].data;
    var series = data['series'];
    var categories = data['categories'];
    InitInvoiceChartData(series, categories);
  });

  // Shipment statistics Chart
  // --------------------------------------------------------------------

  function InitInvoiceChartData(series, categories) {
    console.log('Budget Chart Data:', series, categories);

    const allValues = series.flatMap(s => s.data);
    const maxValue = Math.max(...allValues);
    const yAxisMax = Math.ceil((maxValue * 1.2) / 10000) * 10000;

    const shipmentEl = document.querySelector('#InvoiceStatisticsChart');
    if (!shipmentEl) return;

    const shipmentConfig = {
      series: series,
      chart: {
        height: 320,
        type: 'line',
        stacked: false,
        parentHeightOffset: 0,
        toolbar: { show: false },
        zoom: { enabled: false }
      },
      markers: {
        size: 5,
        colors: [config.colors.white],
        strokeColors: chartColors.line.series2,
        hover: { size: 6 },
        borderRadius: 4
      },
      stroke: {
        curve: 'smooth',
        width: [0, 3],
        lineCap: 'round'
      },
      legend: {
        show: true,
        position: 'bottom',
        markers: {
          width: 8,
          height: 8,
          offsetX: -3
        },
        height: 40,
        itemMargin: {
          horizontal: 10,
          vertical: 0
        },
        fontSize: '15px',
        fontFamily: 'Public Sans',
        fontWeight: 400,
        labels: {
          colors: headingColor,
          useSeriesColors: false
        },
        offsetY: 10
      },
      grid: {
        strokeDashArray: 8,
        borderColor
      },
      colors: [chartColors.line.series1, chartColors.line.series2],
      fill: {
        opacity: [1, 1]
      },
      plotOptions: {
        bar: {
          columnWidth: '30%',
          startingShape: 'rounded',
          endingShape: 'rounded',
          borderRadius: 4
        }
      },
      dataLabels: { enabled: false },
      xaxis: {
        tickAmount: 10,
        categories: categories,
        labels: {
          style: {
            colors: labelColor,
            fontSize: '13px',
            fontWeight: 400
          }
        },
        axisBorder: { show: false },
        axisTicks: { show: false }
      },
      yaxis: {
        min: 0,
        max: yAxisMax,
        tickAmount: 5,
        labels: {
          formatter: function (val) {
            return val.toLocaleString() + ' ₺';
          }
        }
      },
      responsive: [
        {
          breakpoint: 1400,
          options: {
            chart: { height: 320 },
            xaxis: { labels: { style: { fontSize: '10px' } } },
            legend: {
              itemMargin: {
                vertical: 0,
                horizontal: 10
              },
              fontSize: '13px',
              offsetY: 12
            }
          }
        },
        {
          breakpoint: 1025,
          options: {
            chart: { height: 415 },
            plotOptions: { bar: { columnWidth: '50%' } }
          }
        },
        {
          breakpoint: 982,
          options: { plotOptions: { bar: { columnWidth: '30%' } } }
        },
        {
          breakpoint: 480,
          options: {
            chart: { height: 250 },
            legend: { offsetY: 7 }
          }
        }
      ]
    };

    if (!BudgetChart) {
      BudgetChart = new ApexCharts(shipmentEl, shipmentConfig);
      BudgetChart.render();
    } else {
      BudgetChart.updateOptions(shipmentConfig);
    }
  }

  window.addEventListener('CashFlowChartUpdated', function (event) {
    var data = event.detail[0];
    console.log(data);
    var series = data['series'];
    var categories = data['categories'];
    InitCashFlowChartUpdated(series, categories);
  });
  function InitCashFlowChartUpdated(series, categories) {
    console.log('Cash Flow Chart Data:', series, categories);

    // Calculate max value for Y-axis scaling

    const chartEl = document.querySelector('#CashFlowChart');
    if (!chartEl) return;

    // Custom colors for better financial visualization
    const customColors = {
      rentIncome: '#4CAF50',
      otherIncome: '#81C784',
      expenses: '#F44336',
      actualRent: '#2E7D32',
      actualOther: '#66BB6A',
      actualExpenses: '#C62828'
    };

    const chartConfig = {
      series: series,
      chart: {
        height: 350,
        type: 'line',
        stacked: false,
        parentHeightOffset: 0,
        toolbar: { show: true },
        zoom: { enabled: false }
      },
      markers: {
        size: 5,
        strokeWidth: 2,
        hover: { size: 7 }
      },
      stroke: {
        curve: 'smooth',
        width: [3, 3, 3, 2, 2, 2], // Thicker for projections, thinner for actuals
        dashArray: [0, 0, 0, 5, 5, 5] // Solid for projections, dashed for actuals
      },
      legend: {
        show: true,
        position: 'bottom',
        horizontalAlign: 'center',
        markers: {
          width: 10,
          height: 10,
          radius: 4,
          offsetX: -3
        },
        itemMargin: {
          horizontal: 15,
          vertical: 5
        },
        fontSize: '13px',
        fontFamily: 'Public Sans',
        fontWeight: 500,
        onItemClick: {
          toggleDataSeries: true
        },
        onItemHover: {
          highlightDataSeries: true
        }
      },
      grid: {
        borderColor: borderColor,
        strokeDashArray: 5,
        yaxis: {
          lines: {
            show: true
          }
        },
        padding: {
          top: 0,
          right: 20,
          bottom: 0,
          left: 20
        }
      },
      colors: [
        customColors.rentIncome,
        customColors.otherIncome,
        customColors.expenses,
        customColors.actualRent,
        customColors.actualOther,
        customColors.actualExpenses
      ],
      plotOptions: {
        bar: {
          columnWidth: '25%',
          borderRadius: 6,
          dataLabels: {
            position: 'top'
          }
        }
      },
      dataLabels: {
        enabled: false
      },
      xaxis: {
        type: 'category',
        categories: categories,
        tickAmount: 6,
        labels: {
          style: {
            colors: labelColor,
            fontSize: '12px',
            fontFamily: 'Public Sans',
            fontWeight: 500
          },
          formatter: function (value) {
            return value; // Or format date differently if needed
          }
        },
        axisBorder: {
          show: true,
          color: borderColor
        },
        axisTicks: {
          show: true,
          color: borderColor
        },
        tooltip: {
          enabled: false
        }
      },
      yaxis: {
        tickAmount: 6,
        labels: {
          formatter: function (val) {
            return val.toLocaleString() + ' ₺';
          },
          style: {
            colors: labelColor,
            fontSize: '12px',
            fontFamily: 'Public Sans'
          }
        },
        axisBorder: {
          show: true,
          color: borderColor
        }
      },
      tooltip: {
        shared: true,
        intersect: false,
        y: {
          formatter: function (val) {
            return val.toLocaleString() + ' ₺';
          }
        },
        custom: function ({ series, seriesIndex, dataPointIndex, w }) {
          const month = w.globals.categoryLabels[dataPointIndex];
          let html = `<div class="apexcharts-tooltip-title">${month}</div>`;

          // Group related series together
          const groups = {
            Projected: [0, 1, 2],
            Actual: [3, 4, 5]
          };

          Object.entries(groups).forEach(([groupName, seriesIndices]) => {
            let groupTotal = 0;
            let groupItems = '';

            seriesIndices.forEach(i => {
              if (w.config.series[i] && series[i][dataPointIndex] !== undefined) {
                const value = series[i][dataPointIndex];
                groupTotal += value;
                groupItems += `
                                <div class="tooltip-item">
                                    <span class="tooltip-marker" style="background:${w.config.colors[i]}"></span>
                                    <span class="tooltip-label">${w.config.series[i].name}:</span>
                                    <span class="tooltip-value">${value.toLocaleString()} ₺</span>
                                </div>
                            `;
              }
            });

            if (groupItems) {
              html += `
                            <div class="tooltip-group">
                                <div class="tooltip-group-header">${groupName}</div>
                                ${groupItems}
                                <div class="tooltip-group-total">
                                    <span>Total:</span>
                                    <span>${groupTotal.toLocaleString()} ₺</span>
                                </div>
                            </div>
                        `;
            }
          });

          // Calculate net cash flow
          const projectedIncome = (series[0][dataPointIndex] || 0) + (series[1][dataPointIndex] || 0);
          const projectedExpenses = series[2][dataPointIndex] || 0;
          const netCashFlow = projectedIncome - projectedExpenses;

          html += `
                    <div class="tooltip-net">
                        <span>Projected Net Cash Flow:</span>
                        <span style="color:${netCashFlow >= 0 ? customColors.rentIncome : customColors.expenses}">
                            ${netCashFlow.toLocaleString()} ₺
                        </span>
                    </div>
                `;

          return html;
        }
      },
      responsive: [
        {
          breakpoint: 1440,
          options: {
            chart: { height: 320 },
            legend: {
              itemMargin: {
                vertical: 0,
                horizontal: 10
              },
              fontSize: '12px'
            }
          }
        },
        {
          breakpoint: 1200,
          options: {
            chart: { height: 300 },
            legend: {
              position: 'bottom',
              horizontalAlign: 'center'
            }
          }
        },
        {
          breakpoint: 992,
          options: {
            chart: { height: 350 },
            plotOptions: {
              bar: {
                columnWidth: '35%'
              }
            }
          }
        },
        {
          breakpoint: 768,
          options: {
            chart: { height: 300 },
            legend: {
              fontSize: '11px',
              itemMargin: {
                horizontal: 8,
                vertical: 4
              }
            }
          }
        },
        {
          breakpoint: 576,
          options: {
            chart: { height: 280 },
            legend: {
              fontSize: '10px',
              offsetY: 5
            },
            stroke: {
              width: [2, 2, 2, 1, 1, 1]
            }
          }
        }
      ]
    };

    // Initialize or update chart
    if (typeof CashFlowChart === 'undefined') {
      CashFlowChart = new ApexCharts(chartEl, chartConfig);
      CashFlowChart.render();
    } else {
      CashFlowChart.updateOptions(chartConfig);
    }
  }
  window.addEventListener('PropertyTypeChartDataUpdated', function (event) {
    if (propertyTypeChart) propertyTypeChart.destroy();

    var data = event.detail[0].data;

    propertyTypeChart = new ApexCharts(document.querySelector('#property-type-chart'), {
      chart: {
        type: 'bar',
        stacked: true,
        height: 400
      },
      colors: ['#34A853', '#FB8C00'], // Green for count, Orange for rent
      series: data.series,
      xaxis: {
        categories: data.labels
      },
      plotOptions: {
        bar: {
          horizontal: false,
          columnWidth: '50%',
          endingShape: 'rounded'
        }
      },
      dataLabels: {
        enabled: true
      },
      tooltip: {
        y: {
          formatter: function (val, opts) {
            let seriesName = opts.seriesIndex === 0 ? ' properties' : ' rent';
            return val + seriesName;
          }
        }
      }
    });

    propertyTypeChart.render();
  });
  window.addEventListener('MonthlyExpenseChartDataUpdated', function (event) {
    const data = event.detail[0].data;
    InitMonthlyExpenseChart(data.amounts, data.categories);
  });
  function InitMonthlyExpenseChart(series, categories) {
    const chartEl = document.querySelector('#totalExpensesChart');
    const chartConfig = {
      series: [
        {
          name: 'Total Expenses',
          data: series
        }
      ],
      chart: {
        height: 320,
        type: 'line',
        toolbar: { show: false }
      },
      stroke: {
        curve: 'smooth',
        width: 3
      },
      xaxis: {
        categories: categories,
        labels: {
          style: {
            colors: '#6E6B7B',
            fontSize: '13px'
          }
        }
      },
      yaxis: {
        labels: {
          formatter: val => val.toLocaleString() + ' ₺'
        }
      },
      colors: ['#FF6F61'],
      markers: {
        size: 5
      },
      tooltip: {
        y: {
          formatter: val => val.toLocaleString() + ' ₺'
        }
      }
    };

    if (chartEl) {
      new ApexCharts(chartEl, chartConfig).render();
    }
  }
  window.addEventListener('filtersApplied', function (event) {
    const data = event.detail[0];
    console.log(data);
    let table = window.LaravelDataTables['budget-table'];
    if (table) {
      // Stringify the filters object before storing
      window.filters = JSON.parse(JSON.stringify(data.filters));

      // Force a full reload
      table.ajax.reload();
    }
    table = window.LaravelDataTables['charity-table'];
    if (table) {
      // Stringify the filters object before storing
      window.filters = JSON.parse(JSON.stringify(data.filters));

      // Force a full reload
      table.ajax.reload();
    }
  });
})();
